/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * SAX events representing a newsletter category
 *
 */
public class CategoryGenerator extends ServiceableGenerator
{
    /** The object resolver */
    protected AmetysObjectResolver _resolver;
    /** The category provider manager */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "categories");
        
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        @SuppressWarnings("unchecked")
        List<String> categoryIDs = (List<String>) jsParameters.get("id");
       
        saxCategories(categoryIDs);
        
        XMLUtils.endElement(contentHandler, "categories");
        contentHandler.endDocument();
        
    }
    
    /**
     * SAX categories
     * @param categoryIDs The category ids
     * @throws SAXException if an error occurred while SAXing
     */
    protected void saxCategories (List<String> categoryIDs) throws SAXException
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        
        for (String categoryID : categoryIDs)
        {
            for (String id : ids)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(id);
                if (provider.hasCategory(categoryID))
                {
                    Category category = provider.getCategory(categoryID);
                    saxCategory (provider, category, false, false);
                }
            }
        }
    }
    /**
     * SAX a category
     * @param provider The category provider
     * @param category The category to SAX
     * @param children true to SAX category sub-categories
     * @param newsletter true to SAX the newsletter linked with this category
     * @throws SAXException if an error occurred while SAXing
     */
    protected void saxCategory(CategoryProvider provider, Category category, boolean children, boolean newsletter) throws SAXException
    {
        String id = category.getId();
        I18nizableText title = category.getTitle();
        I18nizableText description = category.getDescription();
        String template = category.getTemplate();
        String siteName = category.getSiteName();
        String lang = category.getLang();
        Collection<String> automaticIds = provider.getAutomaticIds(id);
        
        AttributesImpl attr = new AttributesImpl();
        attr.addAttribute("", "id", "id", "CDATA", category.getId());
        attr.addAttribute("", "parentId", "parentId", "CDATA", category.getParentId());
        attr.addAttribute("", "name", "name", "CDATA", category.getName());
        attr.addAttribute("", "siteName", "siteName", "CDATA", category.getSiteName());
        attr.addAttribute("", "lang", "lang", "CDATA", category.getLang());
        
        if (provider.hasChildren(id) || provider.hasNewsletters(id, siteName, lang))
        {
            attr.addAttribute("", "hasChild", "hasChild", "CDATA", "true");
        }
        
        attr.addAttribute("", "mode", "mode", "CDATA", provider.isWritable() ? "write" : "read");
        attr.addCDATAAttribute("automaticIds", StringUtils.join(automaticIds, ','));
        
        XMLUtils.startElement(contentHandler, "category", attr);
        
        XMLUtils.startElement(contentHandler, "title");
        title.toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "title");

        if (description != null)
        {
            XMLUtils.startElement(contentHandler, "description");
            description.toSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "description");
        }
        
        XMLUtils.createElement(contentHandler, "template", template != null ? template : "");
        
        if (children)
        {
            List<Category> childCategories = provider.getCategories(category.getId());
            for (Category child : childCategories)
            {
                saxCategory(provider, child, false, false);
            }
        }
        
        if (newsletter)
        {
            // SAX newsletters
            AmetysObjectIterable<Content> contents = provider.getNewsletters(id, siteName, lang);
            saxNewsletters(contents);
        }
        
        XMLUtils.endElement(contentHandler, "category");
    }
    
    /**
     * SAX the newsletters
     * @param newsletters The newsletters to SAX
     * @throws SAXException If an error occurred while SAXing
     */
    protected void saxNewsletters (AmetysObjectIterable<Content> newsletters) throws SAXException
    {
        for (Content content : newsletters)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addAttribute("", "id", "id", "CDATA", content.getId());
            attrs.addAttribute("", "name", "name", "CDATA", content.getName());
            attrs.addAttribute("", "title", "title", "CDATA", content.getTitle());

            boolean automatic = content.getInternalDataHolder().getValue("automatic", false);
            attrs.addAttribute("", "automatic", "automatic", "CDATA", String.valueOf(automatic));
            
            XMLUtils.createElement(contentHandler, "newsletter", attrs);
        }
    }
}
