/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

/**
 * This action subscribes an email address to a newsletter
 *
 */
public class SubscribeAction extends AbstractSubscribeAction
{    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        boolean subscribe = request.getParameter("subscribe") != null;
        
        String email = request.getParameter("email");
        result.put("email", email);
        
        ZoneItem currentZoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        String zoneItemId = request.getParameter("zoneitem-id");

        if (subscribe && (zoneItemId == null || currentZoneItem.getId().equals(zoneItemId))) // Handle zoneitem id null for legacy purpose
        {
            try
            {
                String siteName = request.getParameter("siteName");
                String[] categories = request.getParameterValues("category");
                
                // Validate email
                if (!_validEmail(email))
                {
                    result.put("msg", "invalid-email");
                    return result;
                }
                

                // Validate captcha
                SitemapElement sitemapElement;
                if (currentZoneItem == null)
                {
                    sitemapElement = _resolver.resolveById(request.getParameter("page-id"));
                }
                else
                {
                    sitemapElement = currentZoneItem.getZone().getSitemapElement();
                }
                if (!_validCaptcha(request, sitemapElement))
                {
                    result.put("msg", "invalid-captcha");
                    return result;
                }
                
                // Validate categories
                if (!_validCategory(categories))
                {
                    result.put("msg", "invalid-category");
                    return result;
                }
                
                
                List<String> subscribeTo = new ArrayList<>();
                List<String> alreadySubscribeTo = new ArrayList<>();
                for (String categoryID : categories)
                {
                    if (_subscribersDao.getSubscriber(email, siteName, categoryID) == null)
                    {
                        Category category = _getCategory(categoryID);
                        if (category != null)
                        {
                            Subscriber subscriber = _createSubscritpion(email, siteName, categoryID);
                            _subscribersDao.subscribe(subscriber);
                            
                            getLogger().info("The user with email '" + email + "' subscribed to the newsletter with the token " + subscriber.getToken());
                            
                            subscribeTo.add(categoryID);
                        }
                    }
                    else
                    {
                        alreadySubscribeTo.add(categoryID);
                    }
                }
                
                result.put("msg", "success");
                result.put("subscribeTo", String.join(",", subscribeTo));
                result.put("alreadySubscribeTo", String.join(",", alreadySubscribeTo));
            }
            catch (Exception e)
            {
                result.put("msg", "failure");
                getLogger().error("An error occurred during the subscription for the email '" + email + "'", e);
            }
        }
        return result;
    }
    
    /**
     * Determines if the categories are valid
     * @param categories The categories id
     * @return true if the categories are valid
     */
    protected boolean _validCategory (String[] categories)
    {
        return categories != null && categories.length > 0;
    }
}
