/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.newsletter.daos.SubscribersDAO.UnsubscribeOrigin;

/**
 * Client side element for newsletter subscribers
 */
public class SubscribersClientSideElement extends StaticClientSideElement
{
    private static final Pattern __EMAIL_VALIDATOR = SendMailHelper.EMAIL_VALIDATION;
    
    /** The subscribers DAO. */
    protected SubscribersDAO _subscribersDao;
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryProviderEP;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscribersDao = (SubscribersDAO) smanager.lookup(SubscribersDAO.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    /**
     * Add a list of subscribers to the newsletter category of the specified site
     * @param siteName The site name
     * @param categoryId The category id
     * @param emailsList The list of emails. Emails can be separeted by a new line or a semicolon.
     * @return The result, with the count of subscribed or error emails.
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleSubscribers")
    public Map<String, Object> addSubscribers (String siteName, String categoryId, String emailsList)
    {
        Map<String, Object> result = new HashMap<>();

        // Test if the category exists.
        Category category = _categoryProviderEP.getCategory(categoryId);
        if (category == null)
        {
            return Collections.singletonMap("error", "unknown-category");
        }
        
        // Extract the emails from the import file.
        Collection<String> emails = getEmails(emailsList);
        
        if (!emails.isEmpty())
        {
            // Insert the emails.
            insertSubscribers(emails, categoryId, siteName, result);
        }
        else
        {
            result.put("subscribedCount", "0");
            result.put("existingCount", "0");
            result.put("errorCount", "0");
        }
        
        result.put("success", "true");
        result.put("categoryId", categoryId);
        
        return result;
    }
    
    /**
     * Extract the emails from the field.
     * @param emailsList the list of emails
     * @return a collection of the emails.
     */
    protected Collection<String> getEmails(String emailsList)
    {
        Set<String> emails = new LinkedHashSet<>();
        
        String[] part = emailsList.split("[;\n]");
        for (String line : part)
        {
            String email = StringUtils.trimToEmpty(line);
            if (__EMAIL_VALIDATOR.matcher(email.toLowerCase()).matches() && StringUtils.isNotBlank(email))
            {
                emails.add(email);
            }
            else
            {
                getLogger().warn("Import subscribers email: '" + email + "' is not a valid email; it will be ignored");
            }
        }
        
        return emails;
    }
    

    /**
     * Insert subscribers
     * @param emails the list of emails
     * @param categoryId the id of the newsletter category
     * @param siteName the name of the sites
     * @param result the result map
     */
    protected void insertSubscribers(Collection<String> emails, String categoryId, String siteName, Map<String, Object> result)
    {
        int subscribedCount = 0;
        int existingCount = 0;
        int errorCount = 0;
        
        for (String email : emails)
        {
            if (_subscribersDao.getSubscriber(email, siteName, categoryId) == null)
            {
                Subscriber subscriber = new Subscriber();
                subscriber.setEmail(email);
                subscriber.setSiteName(siteName);
                subscriber.setCategoryId(categoryId);
                subscriber.setSubscribedAt(new Date());
                
                // Generate unique token.
                String token = UUID.randomUUID().toString();
                subscriber.setToken(token);
                
                _subscribersDao.subscribe(subscriber);
                
                if (getLogger().isInfoEnabled())
                {
                    getLogger().info("The user with email '" + email + "' subscribed to the newsletter with the token " + token);
                }
                
                subscribedCount++;
            }
            else
            {
                existingCount++;
            }
        }
        
        result.put("subscribedCount", Integer.toString(subscribedCount));
        result.put("existingCount", Integer.toString(existingCount));
        result.put("errorCount", Integer.toString(errorCount));
    }
    
    
    /**
     * Remove a list of subscribers from the newsletter category of the specified site
     * @param siteName The site name
     * @param categoryId The category id
     * @param emails The list of emails.
     * @return The error message, or an empty map if everything is ok.
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleSubscribers")
    public Map<String, Object> removeSubscribers (String siteName, String categoryId, List<String> emails)
    {
        Map<String, Object> result = new HashMap<>();
    
        // Test if the category exists.
        Category category = _categoryProviderEP.getCategory(categoryId);
        if (category == null)
        {
            return Collections.singletonMap("message", "unknown-category");
        }
        
        if (emails.size() > 0)
        {
            for (String email : emails)
            {
                Subscriber subscriber = _subscribersDao.getSubscriber(email, siteName, categoryId);
                
                if (subscriber != null)
                {
                    String token = subscriber.getToken();
                    
                    _subscribersDao.unsubscribe(token, UnsubscribeOrigin.ADMINISTRATOR);
                    
                    if (getLogger().isInfoEnabled())
                    {
                        getLogger().info("The user with email '" + email + "' unsubscribed from the newsletter in category " + categoryId + " of site " + siteName);
                    }
                }
                else
                {
                    getLogger().error("Impossible to find and remove from the newsletter the subscriber with email '" + email + "' in category " + categoryId + " of site " + siteName);
                }
            }
        }
        
        return result;
    }
}
