/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;

/**
 * SAX events for newsletter subscribers
 *
 */
public class GetSubscribersAction extends ServiceableAction
{
    private SubscribersDAO _subscribersDao;
    private JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _subscribersDao = (SubscribersDAO) serviceManager.lookup(SubscribersDAO.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String categoryId = (String) jsParameters.get("categoryID");
        String siteName = (String) jsParameters.get("siteName");
        int begin = (Integer) jsParameters.getOrDefault("start", 0); // Index of search
        int offset = (Integer) jsParameters.getOrDefault("limit", Integer.MAX_VALUE); // Number of results to SAX
        List<Map<String, Object>> sorts = _getSorts(jsParameters.get("sort"));
        List<Map<String, Object>> subscribersResult = new ArrayList<>();
        
        List<Subscriber> subscribers;
        int total = 0;
        if (siteName != null && categoryId != null)
        {
            subscribers = _subscribersDao.getSubscribers(siteName, categoryId, sorts, begin, offset);
            total = _subscribersDao.getSubscribersCount(siteName, categoryId);
        }
        else
        {
            subscribers = _subscribersDao.getSubscribers(sorts);
            total = subscribers.size();
        }
        
        for (Subscriber subscriber : subscribers)
        {
            subscribersResult.add(subscriberToJSON(subscriber));
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("subscribers", subscribersResult);
        result.put("total", total);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    /**
     * Gets category provider's properties to JSON format
     * @param subscriber the subscriber
     * @return The category provider properties
     */
    protected Map<String, Object> subscriberToJSON (Subscriber subscriber)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("email", subscriber.getEmail());
        infos.put("siteName", subscriber.getSiteName());
        infos.put("category", subscriber.getCategoryId());
        infos.put("subscribedAt", DateUtils.dateToString(subscriber.getSubscribedAt()));
        
        return infos;
    }

    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> _getSorts(Object sortValues)
    {
        if (sortValues != null)
        {
            return (List<Map<String, Object>>) (Object) _jsonUtils.convertJsonToList(sortValues.toString());
        }
        
        return null;
    }
}
