/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.web.repository.site.Site;

/**
 * {@link AmetysObject} for storing tag informations.
 */
public class JCRCategory extends DefaultTraversableAmetysObject<CategoryFactory>
{
    /** Constants for title metadata. */
    private static final String __METADATA_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";
    /** Constants for description metadata. */
    private static final String __METADATA_DESC = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":description";
    /** Constants for template metadata. */
    private static final String __METADATA_TEMPLATE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":template";
    /** Constants for automatic newsletter ids. */
    private static final String __AUTOMATIC_IDS = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":automatic-ids";
    
    /**
     * Creates a {@link JCRCategory}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link CategoryFactory} which creates the AmetysObject.
     */
    public JCRCategory (Node node, String parentPath, CategoryFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__METADATA_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__METADATA_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }
    
    /**
     * Retrieves the description.
     * @return the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getDescription() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__METADATA_DESC).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get description property", e);
        }
    }
    
    /**
     * Set the description.
     * @param description the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDescription(String description) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__METADATA_DESC, description);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set description property", e);
        }
    }

    /**
     * Retrieves the template name.
     * @return the template name.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTemplate() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__METADATA_TEMPLATE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get template property", e);
        }
    }
    
    /**
     * Set the template.
     * @param templateName the template.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTemplate(String templateName) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__METADATA_TEMPLATE, templateName);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set template property", e);
        }
    }
    
    /**
     * Retrieves the template name.
     * @return the template name.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Collection<String> getAutomaticIds() throws AmetysRepositoryException
    {
        try
        {
            List<String> automaticIds = new ArrayList<>();
            
            Value[] values = getNode().getProperty(__AUTOMATIC_IDS).getValues();
            for (Value value : values)
            {
                automaticIds.add(value.getString());
            }
            
            return automaticIds;
        }
        catch (PathNotFoundException e)
        {
            return Collections.emptyList();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get template property", e);
        }
    }
    
    /**
     * Set the template.
     * @param autoIds the template.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setAutomaticIds(Collection<String> autoIds) throws AmetysRepositoryException
    {
        try
        {
            // Convert to array.
            String[] autoIdsArray = autoIds.toArray(new String[autoIds.size()]);
            
            getNode().setProperty(__AUTOMATIC_IDS, autoIdsArray);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set template property", e);
        }
    }
    
    /**
     * Get the site name
     * @return The site name
     */
    public String getSiteName ()
    {
        AmetysObject parent = getParent();
        while (parent != null)
        {
            if (parent instanceof Site)
            {
                return ((Site) parent).getName();
            }
            parent = parent.getParent();
        }
        return null;
    }
    
    /**
     * Returns the language
     * @return the language
     */
    public String getLang ()
    {
        AmetysObject parent = getParent();
        while (parent != null)
        {
            if (parent.getParent().getName().equals("ametys:categories"))
            {
                return parent.getName();
            }
            parent = parent.getParent();
        }
        return null;
    }
}
