/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.daos;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.ibatis.session.RowBounds;
import org.apache.ibatis.session.SqlSession;

import org.ametys.core.datasource.AbstractMyBatisDAO;
import org.ametys.core.trace.ForensicLogger;

/**
 * DAO for accessing newsletters subscribers.
 */
public class SubscribersDAO extends AbstractMyBatisDAO implements ThreadSafe
{
    /** The Avalon role name. */
    public static final String ROLE = SubscribersDAO.class.getName();
    
    private static final Map<String, String> __ORDER_TO_COL_MAPPING = Map.of(
        "email", "Email",
        "siteName", "Site_Name",
        "categoryId", "Category",
        "subscribedAt", "Subscribed_At",
        "token", "Token"
    );

    /**
     * Origin of the unsubscription
     */
    public enum UnsubscribeOrigin
    {
        /** the subscriber to action to unsubscribe */
        SUBSCRIBER,
        /** an administrator of the newsletter unsubscribed the subscriber */
        ADMINISTRATOR,
        /** a data policy removed the subscription */
        DATAPOLICY,
    }
    
    private List<Subscriber> _getSubscribers(Map<String, Object> params)
    {
        return _getSubscribers(params, null);
    }
    
    private List<Subscriber> _getSubscribers(Map<String, Object> params, List<Map<String, Object>> sorts)
    {
        return _getSubscribers(params, sorts, RowBounds.DEFAULT);
    }
    
    private List<Subscriber> _getSubscribers(Map<String, Object> params, List<Map<String, Object>> sorts, RowBounds rowBounds)
    {
        Map<String, Object> queryParams = new HashMap<>(params);
        
        if (sorts != null)
        {
            StringBuilder sortString = new StringBuilder();
            
            for (Map<String, Object> sort : sorts)
            {
                String property = (String) sort.get("property");
                sortString.append(__ORDER_TO_COL_MAPPING.getOrDefault(property, property));
                sortString.append(" ");
                sortString.append(Optional.of("direction").map(sort::get).orElse("ASC"));
                sortString.append(",");
            }
            
            if (!sorts.isEmpty())
            {
                sortString.deleteCharAt(sortString.length() - 1);
                queryParams.put("__order", sortString.toString());
            }
        }
        
        try (SqlSession session = getSession())
        {
            return session.selectList("Subscribers.getSubscribers", queryParams, rowBounds);
        }
    }

    /**
     * Get the whole list for subscribers
     * @return The list for subscribers
     */
    public List<Subscriber> getSubscribers()
    {
        return _getSubscribers(Map.of());
    }
    
    /**
     * Get the whole list for subscribers
     * @param sorts The sorts list
     * @return The list for subscribers
     */
    public List<Subscriber> getSubscribers(List<Map<String, Object>> sorts)
    {
        return _getSubscribers(Map.of(), sorts);
    }
    
    /**
     * Get the subscribers to a newsletter category
     * @param siteName The site name
     * @param categoryId The newsletter category's id
     * @return the subscribers
     */
    public List<Subscriber> getSubscribers (String siteName, String categoryId)
    {
        return _getSubscribers(
                Map.of(
                    "siteName", siteName,
                    "category", categoryId
                )
            );
    }
    
    /**
     * Get the subscribers to a newsletter category
     * @param siteName The site name
     * @param categoryId The newsletter category's id
     * @param sorts The sorts list
     * @param offset The number of results to ignore.
     * @param limit The maximum number of results to return.
     * @return the subscribers
     */
    public List<Subscriber> getSubscribers (String siteName, String categoryId, List<Map<String, Object>> sorts, int offset, int limit)
    {
        return _getSubscribers(
                Map.of(
                    "siteName", siteName,
                    "category", categoryId
                ),
                sorts,
                new RowBounds(offset, limit)
            );
    }
    
    /**
     * Get the subscribers count for a newsletter category
     * @param siteName The site name
     * @param categoryId The newsletter category's id
     * @return the subscribers count
     */
    public int getSubscribersCount (String siteName, String categoryId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("siteName", siteName);
        params.put("category", categoryId);
        
        try (SqlSession session = getSession())
        {
            return (Integer) session.selectOne("Subscribers.getSubscribersCount", params);
        }
    }
    
    /**
     * Get a subscriber to a newsletter category
     * @param email The subscriber email
     * @param siteName The site name
     * @param categoryId The newsletter category's id
     * @return the subscribers
     */
    public Subscriber getSubscriber (String email, String siteName, String categoryId)
    {
        return _getSubscribers(
                Map.of(
                    "email", email,
                    "siteName", siteName,
                    "category", categoryId
                )
            )
            .stream()
            .findFirst()
            .orElse(null);
    }
    
    /**
     * Get a subscriber by his token
     * @param token The user token
     * @return the subscribers
     */
    public Subscriber getSubscriberByToken (String token)
    {
        return _getSubscribers(Map.of("token", token))
            .stream()
            .findFirst()
            .orElse(null);
    }
    
    /**
     * Get the list of subscriptions for a given email and site name.
     * @param email the email.
     * @param siteName the site name.
     * @return the list of subscriptions.
     */
    public List<Subscriber> getSubscriptions(String email, String siteName)
    {
        return _getSubscribers(
                Map.of(
                    "email", email,
                    "siteName", siteName
                )
            );
    }
    
    /**
     * Subscribes to the newsletter
     * @param subscriber The subscriber
     */
    public void subscribe (Subscriber subscriber)
    {
        try (SqlSession session = getSession(true))
        {
            session.insert("Subscribers.subscribe", subscriber);
        }
    }
    
    /**
     * Insert several subscriptions to newsletters.
     * @param subscribers a list of subscribers.
     */
    public void subscribe(Collection<Subscriber> subscribers)
    {
        try (SqlSession session = getSession())
        {
            for (Subscriber subscriber : subscribers)
            {
                session.insert("Subscribers.subscribe", subscriber);
            }
            
            session.commit();
        }
    }
    
    
    /**
     * Insert several subscriptions to newsletters.
     * @param newSubscribers the collection of subscribers to insert.
     * @param removeSubscriptions the collection of subscription tokens to remove.
     * @param origin the origin of the unsubscription
     */
    public void modifySubscriptions(Collection<Subscriber> newSubscribers, Collection<String> removeSubscriptions, UnsubscribeOrigin origin)
    {
        try (SqlSession session = getSession())
        {
            for (Subscriber subscriber : newSubscribers)
            {
                session.insert("Subscribers.subscribe", subscriber);
            }
            
            for (String tokenToRemove : removeSubscriptions)
            {
                Subscriber subscriber = getSubscriberByToken(tokenToRemove);
                if (subscriber != null)
                {
                    session.update("Subscribers.unsubscribe", tokenToRemove);
                    ForensicLogger.info("newsletter.unsubscribe", Map.of("email", subscriber.getEmail(), "siteName", subscriber.getSiteName(), "category", subscriber.getCategoryId(), "origin", origin.name()), null);
                }
            }
            
            session.commit();
        }
    }
    
    /**
     * Unsubscribes to the newsletter
     * @param token The unique token
     * @param origin the origin of the unsubscription
     */
    public void unsubscribe (String token, UnsubscribeOrigin origin)
    {
        try (SqlSession session = getSession(true))
        {
            Subscriber subscriber = getSubscriberByToken(token);
            if (subscriber != null)
            {
                session.update("Subscribers.unsubscribe", token);
                ForensicLogger.info("newsletter.unsubscribe", Map.of("email", subscriber.getEmail(), "siteName", subscriber.getSiteName(), "category", subscriber.getCategoryId(), "origin", origin.name()), null);
            }
        }
    }
    
    /**
     * Empty a category's subscribers.
     * @param categoryId the category to empty.
     * @param siteName the site name.
     */
    public void empty(String categoryId, String siteName)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("categoryId", categoryId);
        params.put("siteName", siteName);
        
        try (SqlSession session = getSession(true))
        {
            session.delete("Subscribers.empty", params);
        }
    }
    
    /**
     * Remove all subscriptions for a subscriber in a given site.
     * @param email the category to empty.
     * @param siteName the site name.
     * @param origin the origin of the unsubscription
     * @return the number of subscription removed
     */
    public int unsubscribe(String email, String siteName, UnsubscribeOrigin origin)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("email", email);
        params.put("siteName", siteName);
        
        try (SqlSession session = getSession(true))
        {
            int result = session.delete("Subscribers.removeSubscriptionsByEmail", params);
            if (result > 0)
            {
                ForensicLogger.info("newsletter.unsubscribe", Map.of("email", email, "siteName", siteName, "origin", origin.name()), null);
            }
            return result;
        }
    }
}
