/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.workflow;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.plugins.newsletter.NewsletterDAO;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.workflow.EnhancedFunction;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.content.jcr.DefaultWebContent;
import org.ametys.web.repository.site.Site;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function for creating a content.
 */
public class SendNewsletterFunction extends AbstractContentWorkflowComponent implements EnhancedFunction, Contextualizable
{
    private SubscribersDAO _subscribersDAO;
    private CategoryProviderExtensionPoint _categoryProviderEP;
    
    private Context _context;
    private NewsletterDAO _newsletterDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _subscribersDAO = (SubscribersDAO) manager.lookup(SubscribersDAO.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _newsletterDAO = (NewsletterDAO) manager.lookup(NewsletterDAO.ROLE);
    }

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        Request request = _getRequest();
        if (request.getParameter("send") == null && request.getAttribute("send") == null)
        {
            // Do not send the newsletter
            return;
        }
        
        DefaultWebContent content = (DefaultWebContent) getContent(transientVars);
        try
        {
            // Subscribers
            String categoryID = content.getInternalDataHolder().getValue("category");
            String siteName = content.getSiteName();
            Site site = content.getSite();

            boolean descending = site.getValue("newsletter-subscription-descending", false, false);
            boolean ascending = site.getValue("newsletter-subscription-ascending", false, false);
            
            List<Subscriber> subscribers = _subscribersDAO.getSubscribers(content.getSiteName(), content.getInternalDataHolder().getValue("category"));
            if (descending)
            {
                subscribers.addAll(_getSubscribersOfParentCategories(categoryID, siteName));
            }
            if (ascending)
            {
                subscribers.addAll(_getSubscribersOfChildCategories(categoryID, siteName));
            }

            Map<String, String> recipients = new HashMap<>();
            for (Subscriber subcriber : subscribers)
            {
                recipients.put(subcriber.getEmail(), subcriber.getToken());
            }            

            _newsletterDAO.sendNewsletter(content, recipients);
            
            content.getInternalDataHolder().setValue("sent", true);
            try
            {
                _newsletterDAO.removeTestNewsletter(content, site);
            }
            catch (RepositoryException e)
            {
                _logger.warn("Unable to remove the temporary test newsletter.", e);
            }
            content.saveChanges();
            
            // Send a google analytics event for every newsletter e-mail sent.
            SendWebAnalyticsEventsEngine sendGaEngine = new SendWebAnalyticsEventsEngine();
            sendGaEngine.initialize(_manager, _context);
            sendGaEngine.parametrize(siteName, content, _newsletterDAO.getCategory(categoryID), recipients.size());
            
            new Thread(sendGaEngine).start();
        }
        catch (IOException e)
        {
            throw new WorkflowException("Unable to send mails !", e);
        }
        catch (ContextException e)
        {
            _logger.warn("Context exception when initializing an engine.", e);
        }
        catch (ServiceException e)
        {
            _logger.warn("Service exception when initializing an engine.", e);
        }
        
    }
    
    /**
     * Get the subscribers of parent categories
     * @param categoryID The category id
     * @param siteName The site name
     * @return the subscribers of parent categories
     */
    protected List<Subscriber> _getSubscribersOfParentCategories (String categoryID, String siteName)
    {
        List<Subscriber> subscribers = new ArrayList<>();
        
        Category category = _newsletterDAO.getCategory(categoryID);
        Category parentCategory = _newsletterDAO.getCategory(category.getParentId());
        while (parentCategory != null)
        {
            subscribers.addAll(_subscribersDAO.getSubscribers(siteName, parentCategory.getId()));
            parentCategory = _newsletterDAO.getCategory(parentCategory.getParentId());
        }
        
        return subscribers;
    }
    
    /**
     * Get the subscribers of child categories
     * @param categoryID The category id
     * @param siteName The site name
     * @return The subscribers of child categories
     */
    protected List<Subscriber> _getSubscribersOfChildCategories (String categoryID, String siteName)
    {
        List<Subscriber> subscribers = new ArrayList<>();
        
        CategoryProvider provider = _getProvider(categoryID);
        
        List<Category> children = provider.getCategories(categoryID);
        for (Category child : children)
        {
            subscribers.addAll(_subscribersDAO.getSubscribers(siteName, child.getId()));
            subscribers.addAll(_getSubscribersOfChildCategories (child.getId(), siteName));
        }
        
        return subscribers;
    }

    /**
     * Retrieve the request from which this component is called.
     * @return the request
     */
    protected Request _getRequest()
    {
        return ContextHelper.getRequest(_context);
    }
    
    /**
     * Get the category provider
     * @param categoryID The category id
     * @return The category
     */
    protected CategoryProvider _getProvider (String categoryID)
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        for (String id : ids)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            if (provider.hasCategory(categoryID))
            {
                return provider;
            }
        }
        
        return null;
    }

    @Override
    public FunctionType getFunctionExecType()
    {
        return FunctionType.POST;
    }

    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.newsletter", "PLUGINS_NEWSLETTER_SEND_NEWSLETTER_FUNCTION_LABEL");
    }
}
