/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.actions;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * This action export the XSL URLs of the view and template to use
 */
public class GetNewsletterTemplateAction extends ServiceableAction
{
    private CategoryProviderExtensionPoint _categoryProviderEP;
    
    private I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        boolean textMode = parameters.getParameterAsBoolean("text-mode", false);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        WebContent content = (WebContent) request.getAttribute(Content.class.getName());
        
        String categoryID = content.getInternalDataHolder().getValue("category");
        
        Set<String> providerIDs = _categoryProviderEP.getExtensionsIds();
        for (String providerID : providerIDs)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(providerID);
            if (provider.hasCategory(categoryID))
            {
                Category category = provider.getCategory(categoryID);
                String templateId = category.getTemplate();
                
                String templateUrl = "skin://newsletter/" + templateId + "/stylesheets/" + (textMode ? "template-text.xsl" : "template.xsl");
                result.put("template", templateUrl);
                
                String categoryTitle = _i18nUtils.translate(category.getTitle(), category.getLang());
                
                result.put("categoryId", categoryID);
                result.put("categoryTitle", categoryTitle);
            }
        }
        
        long num = content.getValue("newsletter-number", false, -1L);
        LocalDate date = content.getValue("newsletter-date");
        
        result.put("title", content.getTitle());
        if (num > -1)
        {
            result.put("number", Long.toString(num));
        }
        if (date != null)
        {
            result.put("date", DateTimeFormatter.ISO_LOCAL_DATE.format(date.atStartOfDay(ZoneId.systemDefault())));
        }
        
        result.put("contentLanguage", content.getLanguage());
        
        Site site = content.getSite();
        if (site != null)
        {
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
            
            Sitemap sitemap = site.getSitemap(content.getLanguage());
            if (sitemap != null)
            {
                request.setAttribute(WebConstants.REQUEST_ATTR_SITEMAP, sitemap);
            }
        }

        return result;
    }
}
