/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.userpref;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableInt;

import org.ametys.core.model.type.ModelItemTypeExtensionPoint;
import org.ametys.core.userpref.UserPreference;
import org.ametys.core.userpref.UserPreferenceProvider;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.web.userpref.FOUserPreferencesConstants;

/**
 * Provides all newsletter categories as boolean user preferences.
 * Used to build a general subscription service, with one checkbox by category.
 */
public class NewsletterUserPreferenceProvider extends AbstractLogEnabled implements UserPreferenceProvider, Serviceable
{
    /** The newsletter pref group. */
    protected static final I18nizableText _NEWSLETTER_PREF_GROUP = new I18nizableText("plugin.newsletter", "PLUGINS_NEWSLETTER_USER_PREFERENCES_GROUP");
    
    /** The newsletter preferences storage component role. */
    protected static final String _NEWSLETTER_PREF_STORAGE_COMPONENT = "org.ametys.plugins.newsletter.userpref.NewsletterUserPreferencesStorage";
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryEP;
    
    /** The user preference types extension point */
    protected ModelItemTypeExtensionPoint _userPreferenceTypeExtensionPoint;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _categoryEP = (CategoryProviderExtensionPoint) serviceManager.lookup(CategoryProviderExtensionPoint.ROLE);
        _userPreferenceTypeExtensionPoint = (ModelItemTypeExtensionPoint) serviceManager.lookup(ModelItemTypeExtensionPoint.ROLE_USER_PREFERENCES);
    }
    
    @Override
    public Collection<UserPreference> getPreferences(Map<String, String> contextVars)
    {
        Set<UserPreference> prefs = new HashSet<>();
        
        String siteName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_SITENAME);
        String sitemapName = contextVars.get(FOUserPreferencesConstants.CONTEXT_VAR_LANGUAGE);
        
        if (StringUtils.isNotEmpty(siteName) && StringUtils.isNotEmpty(sitemapName))
        {
            for (String categoryProviderId : _categoryEP.getExtensionsIds())
            {
                CategoryProvider categoryProvider = _categoryEP.getExtension(categoryProviderId);
            
                MutableInt index = new MutableInt(1);
                List<Category> categories = categoryProvider.getCategories(siteName, sitemapName);
                
                fillUserPreferences(prefs, categories, categoryProvider, index);
            }
        }
        
        return prefs;
    }
    
    /**
     * Create a {@link UserPreference} per newsletter category, recursively.
     * @param prefs the user preferences Set to fill.
     * @param categories the categories to work on.
     * @param categoryProvider the category provider.
     * @param index the current user preference index, to increase while processing categories.
     */
    protected void fillUserPreferences(Set<UserPreference> prefs, List<Category> categories, CategoryProvider categoryProvider, MutableInt index)
    {
        for (Category category : categories)
        {
            // Create the user preference object, add it to the set and increment the preferences index.
            prefs.add(getCategoryUserPref(category, index.intValue()));
            index.increment();
            
            // Process the child categories.
            List<Category> subCategories = categoryProvider.getCategories(category.getId());
            fillUserPreferences(prefs, subCategories, categoryProvider, index);
        }
    }

    /**
     * Get a boolean {@link UserPreference} representing a category.
     * @param category the category.
     * @param index the category index in the extension point.
     * @return a boolean {@link UserPreference} representing the category.
     */
    protected UserPreference getCategoryUserPref(Category category, int index)
    {
        UserPreference pref = new UserPreference();
        
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        params.put("category", category.getTitle());
        I18nizableText label = new I18nizableText("plugin.newsletter", "PLUGINS_NEWSLETTER_USER_PREFERENCES_LABEL", params);
        
        pref.setName(category.getId());
        pref.setLabel(label);
        pref.setDescription(category.getDescription());
        pref.setType(_userPreferenceTypeExtensionPoint.getExtension(ModelItemTypeConstants.BOOLEAN_TYPE_ID));
        pref.setPluginName("newsletter");
        pref.setManagerRole(_NEWSLETTER_PREF_STORAGE_COMPONENT);
        
        pref.setDisplayGroup(_NEWSLETTER_PREF_GROUP);
        pref.setPosition(index);
        
        return pref;
    }
    
}
