/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.workflow;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.SaxBuffer;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.sax.ContentHandlerProxy;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLFilterImpl;

import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.newsletter.auto.AutomaticNewsletterFilterResult;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.workflow.CreateContentFunction;

import com.opensymphony.workflow.WorkflowException;

/**
 * OSWorkflow function for creating a content.
 */
public class CreateNewsletterFunction extends CreateContentFunction implements Initializable, Contextualizable
{
    
    /** Newsletter category key. */
    public static final String NEWSLETTER_CATEGORY_KEY = CreateNewsletterFunction.class.getName() + "$category";
    /** Newsletter number key. */
    public static final String NEWSLETTER_NUMBER_KEY = CreateNewsletterFunction.class.getName() + "$number";
    /** Newsletter date key. */
    public static final String NEWSLETTER_DATE_KEY = CreateNewsletterFunction.class.getName() + "$date";
    /** Newsletter automatic property key. */
    public static final String NEWSLETTER_IS_AUTOMATIC_KEY = CreateNewsletterFunction.class.getName() + "$isAutomatic";
    /** Key for "process auto sections". */
    public static final String NEWSLETTER_PROCESS_AUTO_SECTIONS_KEY = CreateNewsletterFunction.class.getName() + "$processAutoSections";
    /** Key for content ID map when processing auto sections. */
    public static final String NEWSLETTER_CONTENT_ID_MAP_KEY = CreateNewsletterFunction.class.getName() + "$contentIds";
    
    /** The date format. */
    public static final DateFormat NEWSLETTER_DATE_FORMAT = new SimpleDateFormat("yyyyMMdd");
    
    /** The list of attributes to ignore. */
    protected static final Set<String> _IGNORE_ATTRS = new HashSet<>();
    static
    {
        _IGNORE_ATTRS.add("auto-newsletter-ignore");
        _IGNORE_ATTRS.add("auto-newsletter-ignore-if-empty");
        _IGNORE_ATTRS.add("auto-newsletter-insert-filter");
        _IGNORE_ATTRS.add("auto-newsletter-insert-level");
    }

    /** The defaut content insertion level. */
    protected static final String _DEFAULT_LEVEL = "3";
    
    /**
     * The Avalon context
     */
    protected Context _context;
    
    private SourceResolver _sourceResolver;
    private CategoryProviderExtensionPoint _categoryProviderEP;
    
    // Transformation objects.
    private TransformerFactory _transformerFactory;
    private Properties _transformerProperties;
    private SAXParserFactory _saxParserFactory;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        // Initialize transformation objects.
        _transformerFactory = TransformerFactory.newInstance();
        
        _transformerProperties = new Properties();
        _transformerProperties.put(OutputKeys.METHOD, "xml");
        _transformerProperties.put(OutputKeys.ENCODING, "UTF-8");
        _transformerProperties.put(OutputKeys.OMIT_XML_DECLARATION, "yes");
        
        _saxParserFactory = SAXParserFactory.newInstance();
        _saxParserFactory.setNamespaceAware(true);
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.newsletter", "PLUGINS_NEWSLETTER_CREATE_NEWSLETTER_FUNCTION_LABEL");
    }
    
    @Override
    protected void _populateContent(Map transientVars, ModifiableContent content) throws WorkflowException
    {
        super._populateContent(transientVars, content);
        
        String category = (String) transientVars.get(NEWSLETTER_CATEGORY_KEY);
        if (category == null)
        {
            throw new WorkflowException("Missing category");
        }
        
        Long number = (Long) transientVars.get(NEWSLETTER_NUMBER_KEY);
        Date date = (Date) transientVars.get(NEWSLETTER_DATE_KEY);
        boolean isAutomatic = "true".equals(transientVars.get(NEWSLETTER_IS_AUTOMATIC_KEY));
        boolean processAutoSections = "true".equals(transientVars.get(NEWSLETTER_PROCESS_AUTO_SECTIONS_KEY));
        @SuppressWarnings("unchecked")
        Map<String, AutomaticNewsletterFilterResult> filterResults = (Map<String, AutomaticNewsletterFilterResult>) transientVars.get(NEWSLETTER_CONTENT_ID_MAP_KEY);
        
        if (processAutoSections && filterResults == null)
        {
            throw new WorkflowException("Content ID map must not be null if processing automatic sections.");
        }
        

        ModifiableModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
        internalDataHolder.setValue("category", category);
        internalDataHolder.setValue("automatic", isAutomatic);
        
        if (number != null)
        {
            content.setValue("newsletter-number", number);
        }
        if (date != null)
        {
            content.setValue("newsletter-date", DateUtils.asLocalDate(date));
        }
        
        String siteName = (String) transientVars.get(SITE_KEY);
        Site site = _siteManager.getSite(siteName);
        
        // Initialize the content from the model
        _initContentRichText (content, site.getSkinId(), category, processAutoSections, filterResults);
    }
    
    private void _initContentRichText (ModifiableContent content, String skinId, String categoryID, boolean processAutoSections, Map<String, AutomaticNewsletterFilterResult> filterResults) throws WorkflowException
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        for (String id : ids)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            if (provider.hasCategory(categoryID))
            {
                Category category = provider.getCategory(categoryID);
                String templateId = category.getTemplate();
                
                if (templateId == null)
                {
                    throw new WorkflowException ("The template can not be null");
                }
                
                try
                {
                    String text = _getContent (skinId, templateId);
                    
                    // Process automatic newsletter tags.
                    String processedText = _processAutoTags(text, processAutoSections, filterResults);

                    ElementType<RichText> richTextType = ((ElementDefinition) content.getDefinition("content")).getType();
                    
                    DataContext dataContext = RepositoryDataContext.newInstance()
                                                                   .withObject(content)
                                                                   .withDataPath("content");
                    
                    content.setValue("content", richTextType.fromJSONForClient(processedText, dataContext));
                    content.saveChanges();
                }
                catch (IOException e)
                {
                    throw new WorkflowException("Unable to transform rich text", e);
                }
            }
        }
    }
    
    private String _processAutoTags(String text, boolean processAutoSections, Map<String, AutomaticNewsletterFilterResult> filterResults) throws WorkflowException
    {
        StringReader in = new StringReader(text);
        ByteArrayOutputStream baos = new ByteArrayOutputStream(1024);
        
        try
        {
            Transformer transformer = _transformerFactory.newTransformer();
            transformer.setOutputProperties(_transformerProperties);
            XMLReader xmlReader = _saxParserFactory.newSAXParser().getXMLReader();
            
            NewsletterFilter newsletterFilter = new NewsletterFilter(xmlReader, _sourceResolver, processAutoSections, filterResults);
            SAXSource transformSource = new SAXSource(newsletterFilter, new InputSource(in));
            
            transformer.transform(transformSource, new StreamResult(baos));
            
            return baos.toString("UTF-8");
        }
        catch (TransformerException e)
        {
            throw new WorkflowException("Transformer exception.", e);
        }
        catch (SAXException e)
        {
            throw new WorkflowException("SAX exception.", e);
        }
        catch (ParserConfigurationException e)
        {
            throw new WorkflowException("SAX exception.", e);
        }
        catch (UnsupportedEncodingException e)
        {
            throw new WorkflowException("Unsupported encoding.", e);
        }
    }
    
    private String _getContent (String skinId, String templateId) throws IOException, WorkflowException
    {
        SitemapSource src = null;
        Request request = ContextHelper.getRequest(_context);
        if (request == null)
        {
            throw new WorkflowException("Unable to get the request");
        }
        
        try
        {
            request.setAttribute("skin", skinId);
            src = (SitemapSource) _sourceResolver.resolveURI("cocoon://_plugins/newsletter/" + templateId + "/model.xml");
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }
    
    /**
     * Automatic newsletter filter.
     */
    protected class NewsletterFilter extends XMLFilterImpl
    {
        
        private SourceResolver _srcResolver;
        
        private boolean _processAutoSections;
        
        private Map<String, AutomaticNewsletterFilterResult> _filterResults;
        
        private boolean _ignore;
        
        private boolean _ignoreNextLevel;
        
        private int _ignoreDepth;
        
        /**
         * Constructor.
         * @param xmlReader the parent XML reader.
         * @param sourceResolver the source resolver.
         * @param filterResults the filter results.
         * @param processAutoSections true to process auto sections, false to ignore them.
         */
        public NewsletterFilter(XMLReader xmlReader, SourceResolver sourceResolver, boolean processAutoSections, Map<String, AutomaticNewsletterFilterResult> filterResults)
        {
            super(xmlReader);
            _srcResolver = sourceResolver;
            _processAutoSections = processAutoSections;
            _filterResults = filterResults;
        }
        
        @Override
        public void startDocument() throws SAXException
        {
            super.startDocument();
            _ignore = false;
            _ignoreDepth = 0;
        }
        
        @Override
        public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException
        {
            String ignore = atts.getValue("auto-newsletter-ignore");
            String ignoreIfEmpty = atts.getValue("auto-newsletter-ignore-if-empty");
            String insertFilter = atts.getValue("auto-newsletter-insert-filter");
            String insertFilterLevel = atts.getValue("auto-newsletter-insert-level");
            
            if (_ignoreNextLevel)
            {
                _ignoreNextLevel = false;
                _ignore = true;
                _ignoreDepth = 0;
            }
            
            if (StringUtils.isNotEmpty(ignore) || StringUtils.isNotEmpty(ignoreIfEmpty) || StringUtils.isNotEmpty(insertFilter) || StringUtils.isNotEmpty(insertFilterLevel))
            {
                AttributesImpl newAtts = new AttributesImpl();
                
                // Copy all attributes except auto newsletter ones.
                _copyStartElementAttributes(atts, newAtts);
                
                SaxBuffer saxBuffer = null;
                
                if (_processAutoSections && !_ignore)
                {
                    if ("true".equals(ignore))
                    {
                        _ignore = true;
                        _ignoreDepth = 0;
                    }
                    else if (StringUtils.isNotEmpty(ignoreIfEmpty))
                    {
                        _handleIgnoreIfEmpty(ignoreIfEmpty);
                    }
                    else if (StringUtils.isNotEmpty(insertFilter))
                    {
                        saxBuffer = _handleInsertFilter(insertFilter, insertFilterLevel, saxBuffer);
                    }
                }
                
                if (!_ignore)
                {
                    super.startElement(uri, localName, qName, newAtts);
                    
                    if (saxBuffer != null)
                    {
                        _ignoreNextLevel = true;
                        
                        saxBuffer.toSAX(getContentHandler());
                    }
                }
                else
                {
                    _ignoreDepth++;
                }
            }
            else
            {
                // No attribute found, no need to transform anything.
                if (!_ignore)
                {
                    super.startElement(uri, localName, qName, atts);
                }
                else
                {
                    _ignoreDepth++;
                }
            }
        }

        private void _handleIgnoreIfEmpty(String ignoreIfEmpty)
        {
            if (!_filterResults.containsKey(ignoreIfEmpty) || !_filterResults.get(ignoreIfEmpty).hasResults())
            {
                _ignore = true;
                _ignoreDepth = 0;
            }
        }

        private SaxBuffer _handleInsertFilter(String insertFilter, String insertFilterLevel, SaxBuffer saxBuffer) throws SAXException
        {
            SaxBuffer modifiedSaxBuffer = saxBuffer;
            
            if (_filterResults.containsKey(insertFilter) && _filterResults.get(insertFilter).hasResults())
            {
                AutomaticNewsletterFilterResult result = _filterResults.get(insertFilter);
                List<String> contentIds = result.getContentIds();
                
                String viewName = result.getViewName();
                String level = StringUtils.defaultIfEmpty(insertFilterLevel, _DEFAULT_LEVEL);
                
                modifiedSaxBuffer = _getFilterContent(contentIds, level, viewName);
            }
            else
            {
                _ignore = true;
                _ignoreDepth = 0;
            }
            
            return modifiedSaxBuffer;
        }

        private void _copyStartElementAttributes(Attributes atts, AttributesImpl newAtts)
        {
            for (int i = 0; i < atts.getLength(); i++)
            {
                String attrName = atts.getLocalName(i);
                if (!_IGNORE_ATTRS.contains(attrName))
                {
                    newAtts.addAttribute(atts.getURI(i), attrName, atts.getQName(i), atts.getType(i), atts.getValue(i));
                }
            }
        }
        
        @Override
        public void endElement(String uri, String localName, String qName) throws SAXException
        {
            if (_ignoreNextLevel)
            {
                _ignoreNextLevel = false;
            }
            
            if (!_ignore)
            {
                super.endElement(uri, localName, qName);
            }
            else
            {
                _ignoreDepth--;
                
                if (_ignoreDepth < 1)
                {
                    _ignore = false;
                }
            }
        }
        
        @Override
        public void characters(char[] ch, int start, int length) throws SAXException
        {
            if (!_ignore && !_ignoreNextLevel)
            {
                super.characters(ch, start, length);
            }
        }
        
        private SaxBuffer _getFilterContent(List<String> contentIds, String level, String viewName) throws SAXException
        {
            SitemapSource src = null;
            Request request = ContextHelper.getRequest(_context);
            if (request == null)
            {
                throw new SAXException("Unable to get the request");
            }
            
            try
            {
                StringBuilder url = new StringBuilder("cocoon://_plugins/web/contents/last-published");
                url.append("?viewName=").append(viewName).append("&level=").append(level);
                for (String id : contentIds)
                {
                    url.append("&contentId=").append(id);
                }
                
                src = (SitemapSource) _srcResolver.resolveURI(url.toString());
                
                SaxBuffer buffer = new SaxBuffer();
                
                // Ignore the root tag
                src.toSAX(new IgnoreRootTagHandler(buffer));
                
                return buffer;
            }
            catch (IOException e)
            {
                throw new SAXException("Error resolving the contents.", e);
            }
            finally
            {
                _srcResolver.release(src);
            }
        }
        
    }
    
    /**
     * Ignore the root tag.
     */
    protected class IgnoreRootTagHandler extends ContentHandlerProxy
    {
        private int _depth;
        
        /**
         * Constructor
         * @param contentHandler the contentHandler to pass SAX events to.
         */
        public IgnoreRootTagHandler(ContentHandler contentHandler)
        {
            super(contentHandler);
        }
        
        @Override
        public void startDocument() throws SAXException
        {
            _depth = 0;
        }
        
        @Override
        public void endDocument() throws SAXException
        {
            // empty method
        }
        
        @Override
        public void startElement(String uri, String loc, String raw, Attributes a) throws SAXException
        {
            _depth++;
            
            if (_depth > 1)
            {
                super.startElement(uri, loc, raw, a);
            }
        }
        
        @Override
        public void endElement(String uri, String loc, String raw) throws SAXException
        {
            if (_depth > 1)
            {
                super.endElement(uri, loc, raw);
            }
            
            _depth--;
        }
    }
}
