/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.content;

import java.util.Arrays;
import java.util.Locale;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.newsletter.NewsletterDAO;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;

/**
 * Generates SAX events for Content, including category, automatic and sent for Newsletters.
 * TODO NEWATTRIBUTEAPI_CONTENT: do not use type implementation but the ModelAwareDataHolder#getInternalValue when this API exist
 */
public class ContentSaxer extends org.ametys.web.content.ContentSaxer
{
    /** Avalon role. */
    public static final String NEWSLETTER_CONTENT_SAXER_ROLE = ContentSaxer.class.getName();
    
    private static final String __CATEGORY_METADATA_NAME = "category";
    private static final String __AUTOMATIC_METADATA_NAME = "automatic";
    private static final String __SENT_METADATA_NAME = "sent";
    
    @Override
    protected void saxBody(Content content, ContentHandler contentHandler, Locale locale, View view, String tagName, boolean saxWorkflowStep, boolean saxWorkflowInfo,
            boolean saxLanguageInfo, String attributesTagName, boolean isEdition, boolean renderDisableValues) throws SAXException
    {
        super.saxBody(content, contentHandler, locale, view, tagName, saxWorkflowStep, saxWorkflowInfo, saxLanguageInfo, attributesTagName, isEdition, renderDisableValues);
        
        if (Arrays.asList(content.getTypes()).contains(NewsletterDAO.__NEWSLETTER_CONTENT_TYPE))
        {
            saxMetadata(content, contentHandler);
        }
    }
    
    /**
     * Generates SAX events for the newsletter's metadata.
     * @param content the newsletter
     * @param contentHandler the ContentHandler receiving SAX events.
     * @throws SAXException if an error occurs during the SAX events generation.
     */
    protected void saxMetadata(Content content, ContentHandler contentHandler) throws SAXException
    {
        ModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
        DataContext context = DataContext.newInstance().withEmptyValues(false);

        internalDataHolder.dataToSAX(contentHandler, __CATEGORY_METADATA_NAME, context);
        internalDataHolder.dataToSAX(contentHandler, __AUTOMATIC_METADATA_NAME, context);
        internalDataHolder.dataToSAX(contentHandler, __SENT_METADATA_NAME, context);
    }
}
