/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO.UnsubscribeOrigin;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * This action subscribes an email address to a newsletter
 *
 */
public class SubscribeAndManageAction extends AbstractSubscribeAction implements Initializable
{

    /** The user manager */
    protected UserManager _userManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private Cache<String, String> _cache;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }

    public void initialize() throws Exception
    {
        _cache = CacheBuilder.newBuilder().expireAfterWrite(60, TimeUnit.MINUTES).build();
    }
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Map<String, Object> result = new HashMap<>();
        
        Map<String, Object> formResult = new HashMap<>();
        result.put("FormResult", formResult);
        
        // Add user to prefill the email address
        _addUser(result, formResult);
        
        ZoneItem currentZoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        String[] categories = currentZoneItem.getServiceParameters().getValue("categories");
        _addCategories(result, categories);

        
        String zoneItemId = request.getParameter("zoneitem-id");
        // Handle zoneItemId id null for subscription coming from links
        if (zoneItemId == null || currentZoneItem.getId().equals(zoneItemId))
        {
            // Compute current step
            
            // If we have a back value, we go back to step 1.
            if (request.getParameter("back") != null)
            {
                _goToStep1(request, formResult);
            }
            else if (request.getParameter("form-token") != null)
            {
                _goToStep3(request, formResult, categories, currentZoneItem);
            }
            else if (request.getParameter("email") != null || request.getParameter("token") != null)
            {
                // (validate email/captcha or token and get the user subscriptions to display them as checkboxes)
                _goToStep2(request, formResult, categories, currentZoneItem);
            }
            else
            {
                // Check if we need captcha, and fill the email if it had one
                _goToStep1(request, formResult);
            }
        }
        else
        {
            // Check if we need captcha, and fill the email if it had one
            _goToStep1(request, formResult);
        }
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }

    private Map<String, Object> _goToStep1(Request request, Map<String, Object> formResult)
    {
        // Check if we need captcha on this page
        Page page = (Page) request.getAttribute(Page.class.getName());
        formResult.put("has-captcha", _pageHelper.isCaptchaRequired(page));
        
        // In case the user already filled an email address, we want to get it back
        String email = request.getParameter("email");
        if (email != null)
        {
            formResult.put("email", request.getParameter("email"));
        }
        
        formResult.put("step", "1");
        
        return formResult;
    }
    
    private Map<String, Object> _goToStep2(Request request, Map<String, Object> formResult, String[] categories, ZoneItem currentZoneItem)
    {
        String email = null;
        String siteName = currentZoneItem.getZone().getSitemapElement().getSiteName();
        
        String token = request.getParameter("token");
        if (StringUtils.isNotEmpty(token))
        {
            // We come from email link
            Subscriber subscriber = _subscribersDao.getSubscriberByToken(token);
            if (subscriber == null || !siteName.equals(subscriber.getSiteName()))
            {
                formResult.put("msg", "invalid-token");
                return _goToStep1(request, formResult);
            }
            email = subscriber.getEmail();
        }
        else
        {
            // We come from form submission
            if (!_validCaptcha(request, currentZoneItem.getZone().getSitemapElement()))
            {
                formResult.put("msg", "invalid-captcha");
                return _goToStep1(request, formResult);
            }
            formResult.put("is-from-form", "true");
            email =  request.getParameter("email");
        }

        formResult.put("email", email);
        formResult.put("siteName", siteName);

        // Validate email
        if (!_validEmail(email))
        {
            formResult.put("msg", "invalid-email");
            return _goToStep1(request, formResult);
        }

        // Get currently subscribe categories
        String finalEmail = email;
        List<String> alreadySubscribeTo = Arrays.stream(categories)
            .filter(categoryId -> _subscribersDao.getSubscriber(finalEmail, siteName, categoryId) != null)
            .collect(Collectors.toList());
        
        formResult.put("alreadySubscribeTo", alreadySubscribeTo);
        
        // Generate an expirable token that will be used to retrieve the user in step 3
        String formToken = UUID.randomUUID().toString();
        _cache.put(formToken, email);
        formResult.put("form-token", formToken);
        formResult.put("step", "2");
        return formResult;
    }

    private Map _goToStep3(Request request, Map<String, Object> formResult, String[] categories, ZoneItem currentZoneItem)
    {
        String siteName = currentZoneItem.getZone().getSitemapElement().getSiteName();
        String formToken = request.getParameter("form-token");
        String email = _cache.getIfPresent(formToken);
        // Check if the token is still valid
        if (!_validEmail(email))
        {
            formResult.put("msg", "token-expired");
            return _goToStep1(request, formResult);
        }
        
        List<Subscriber> newSubscribers = new ArrayList<>();
        List<String> newSubscriberTokens = new ArrayList<>();
        List<String> removeSubscriptions = new ArrayList<>();
        
        for (String categoryId : categories)
        {
            Subscriber subscriber = _subscribersDao.getSubscriber(email, siteName, categoryId);
            boolean wantToSubscribe = StringUtils.equals("true", request.getParameter("category_" + categoryId));
            if (subscriber == null && wantToSubscribe)
            {
                Category category = _getCategory(categoryId);
                if (category != null)
                {
                    subscriber = _createSubscritpion(email, siteName, categoryId);
                    newSubscribers.add(subscriber);
                    newSubscriberTokens.add(subscriber.getToken());
                }
            }
            else if (subscriber != null && !wantToSubscribe)
            {
                removeSubscriptions.add(subscriber.getToken());
            }

        }
        _subscribersDao.modifySubscriptions(newSubscribers, removeSubscriptions, UnsubscribeOrigin.SUBSCRIBER);
        
        getLogger().info("The user with email '" + email + "' subscribed to the newsletters '" + String.join(",", newSubscriberTokens)
            + "' and unsubscribed to the newsletters '" + String.join(",", removeSubscriptions) + "'");
        
        formResult.put("step", "3");
        return formResult;
    }

    private void _addUser(Map<String, Object> result, Map<String, Object> formResult)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        if (userIdentity != null)
        {
            User user = _userManager.getUser(userIdentity);
            
            result.put("user", Map.of("email", user.getEmail()));
            formResult.put("email", user.getEmail());
        }
    }

    private void _addCategories(Map<String, Object> result, String[] categories)
    {
        
        List<Map<String, Object>> categoryObjectMap = Arrays.stream(categories).map(categoryId ->
        {
            Category category = this._getCategory(categoryId);
            Map<String, Object> categoryMap = new HashMap<>();
            categoryMap.put("id", category.getId());
            categoryMap.put("title", category.getTitle());
            categoryMap.put("description", category.getDescription());
            return categoryMap;
        }).collect(Collectors.toList());
        result.put("categories", categoryObjectMap);
    }
}
