/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.user.status.PersonalDataPolicy;
import org.ametys.core.user.status.UserStatusInfo;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.newsletter.daos.SubscribersDAO.UnsubscribeOrigin;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.config.Config;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Personal data policy removing subscription of missing user
 */
public class SubscriberDataPolicy implements PersonalDataPolicy, Serviceable
{
    /** The population context helper */
    protected PopulationContextHelper _populationContextHelper;
    /** The site DAO */
    protected SiteManager _siteManager;
    /** The subscriber DAO */
    protected SubscribersDAO _subscribersDao;
    /** The user manager */
    protected UserManager _userManager;
    private Period _retentionPeriod;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _populationContextHelper = (PopulationContextHelper) manager.lookup(PopulationContextHelper.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _subscribersDao = (SubscribersDAO) manager.lookup(SubscribersDAO.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        Long config = Config.getInstance().<Long>getValue("runtime.data.policy.userpref.retention", false, null);
        // disable the processing if config is negative
        _retentionPeriod = config != null && config >= 0 ? Period.ofMonths(config.intValue()) : null;
    }
    public AnonymizationResult process(UserStatusInfo userStatusInfo)
    {
        if (_retentionPeriod != null && userStatusInfo.getMissingSinceDate().isBefore(ZonedDateTime.now().minus(_retentionPeriod)))
        {
            // remove the subscription from every site where the unknown user does not belong to a user.
            // the subscription from the remaining site will be processed once the matching user will be deleted
            
            long deletedSubscriptions = 0;
            String email = userStatusInfo.getEmail();
            if (StringUtils.isNotBlank(email))
            {
                try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
                {
                    for (Site site: sites)
                    {
                        String siteName = site.getName();
                        Set<String> userPopulationsOnSite = _populationContextHelper.getUserPopulationsOnContexts(Arrays.asList("/sites/" + siteName, "/sites-fo/" + siteName), false, false);
                        try
                        {
                            if (_userManager.getUserByEmail(userPopulationsOnSite, email) == null)
                            {
                                deletedSubscriptions += _subscribersDao.unsubscribe(email, siteName, UnsubscribeOrigin.DATAPOLICY);
                            }
                        }
                        catch (NotUniqueUserException e)
                        {
                            // Do nothing, this email is still matching existing users
                        }
                    }
                }
            }
            
            if (deletedSubscriptions > 0)
            {
                ForensicLogger.info("data.policy.gdpr.remove.newsletter.subscriptions", Map.of("handled", Long.toString(deletedSubscriptions), "identity", userStatusInfo.getUserIdentity()), UserPopulationDAO.SYSTEM_USER_IDENTITY);
                return AnonymizationResult.PROCESSED;
            }
            else
            {
                return AnonymizationResult.NO_DATA;
            }
        }
        else
        {
            return AnonymizationResult.TOO_EARLY;
        }
    }

}
