/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.LifecycleHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.slf4j.LoggerFactory;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.ThreadSafeComponentManager;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;

/**
 * This element finally creates a gallery button with one item per template  
 */
public class TemplatesMenu extends SimpleMenu implements Contextualizable
{
    /** The plugin in which are the templates resources (messages and icons). */
    protected static final String _RESOURCES_PLUGIN = "newsletter";
    
    private static final String GALLERY_ITEM_MANAGER = TemplatesMenu.class.getName() + "$GalleryItemManager";
    private static final String GALLERY_ITEMS = TemplatesMenu.class.getName() + "$GalleryItems";
    
    /** The skins manager */
    protected SkinsManager _skinsManager;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    /** The service manager */
    protected ServiceManager _manager;
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    private Context _context;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _manager = smanager;
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the available templates for pages
     * @param categoryIds The ids of categories
     * @return the list of templates' name
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleCategory")
    public Map<String, Object> getStatus (List<String> categoryIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("categories", new ArrayList<>());
        
        for (String categoryId : categoryIds)
        {
            JCRCategory category = _resolver.resolveById(categoryId);
            
            // Category parameters
            Map<String, Object> categoryParams = getCategoryDefaultParameters(category);
            categoryParams.put("template", category.getTemplate());
            
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> categories = (List<Map<String, Object>>) result.get("categories");
            categories.add(categoryParams);
        }
        
        return result;
    }
    
    /**
     * Get the default category's parameters
     * @param category The category
     * @return The default parameters
     */
    protected Map<String, Object> getCategoryDefaultParameters (JCRCategory category)
    {
        Map<String, Object> categoryParams = new HashMap<>();
        categoryParams.put("id", category.getId());
        categoryParams.put("title", category.getTitle());
        
        return categoryParams;
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        String siteName = (String) contextParameters.get("siteName");
        
        String skinId = _siteManager.getSite(siteName).getSkinId();
        Skin skin = _skinsManager.getSkin(skinId);
        
        if (skin.getTemplates().size() == 0)
        {
            // Hide menu if there is no template
            return new ArrayList<>();
        }
            
        try
        {
            _lazyInitializeTemplateGallery(contextParameters);
            
            return super.getScripts(ignoreRights, contextParameters);
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
        finally
        {
            _getGalleryItemManager().dispose();
        }
    }

    /**
     * Get templates of newletter categories
     * @param categoryIds the ids of categories
     * @return The list of used templates
     */
    public List<String> getTemplate (List<String> categoryIds)
    {
        List<String> templates = new ArrayList<>();
        
        for (String id : categoryIds)
        {
            Category category = _resolver.resolveById(id);
            
            if (!templates.contains(category.getTemplate()))
            {
                templates.add(category.getTemplate());
            }
        }
        return templates;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected ThreadSafeComponentManager<ClientSideElement> _getGalleryItemManager()
    {
        Request request = ContextHelper.getRequest(_context);
        return (ThreadSafeComponentManager<ClientSideElement>) request.getAttribute(GALLERY_ITEM_MANAGER);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected List<GalleryItem> _getGalleryItems()
    {
        Request request = ContextHelper.getRequest(_context);
        return (List<GalleryItem>) request.getAttribute(GALLERY_ITEMS);
    }
    
    private void _lazyInitializeTemplateGallery(Map<String, Object> contextualParameters) throws ConfigurationException
    {
        String siteName = (String) contextualParameters.get("siteName");
        
        Request request = ContextHelper.getRequest(_context);
        
        @SuppressWarnings("unchecked")
        ThreadSafeComponentManager<ClientSideElement> oldGalleryItemManagerInSameRequest = (ThreadSafeComponentManager<ClientSideElement>) request.getAttribute(GALLERY_ITEM_MANAGER);
        if (oldGalleryItemManagerInSameRequest != null)
        {
            oldGalleryItemManagerInSameRequest.dispose();
        }
        
        ThreadSafeComponentManager<ClientSideElement> galleryItemManager = new ThreadSafeComponentManager<>();
        galleryItemManager.setLogger(LoggerFactory.getLogger("cms.plugin.threadsafecomponent"));
        galleryItemManager.service(_smanager);
        request.setAttribute(GALLERY_ITEM_MANAGER, galleryItemManager);
        
        List<GalleryItem> galleryItems = new ArrayList<>();
        request.setAttribute(GALLERY_ITEMS, galleryItems);
        
        Site site = _siteManager.getSite(siteName);
        String skinId = site.getSkinId();
        
        Set<String> templatesId = new HashSet<>();
        TraversableSource dir;
        try
        {
            dir = (TraversableSource) _sourceResolver.resolveURI("skin:" + skinId + "://newsletter");
            if (dir.exists())
            {                
                for (TraversableSource child : (Collection<TraversableSource>) dir.getChildren())
                {
                    if (_templateExists(skinId, child.getName()))
                    {
                        templatesId.add(child.getName());
                    }
                }
            }
        }
        catch (SourceNotFoundException e)
        {
            // Silently ignore
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Cannot find template for the skin " + skinId, e);
        }
        
        if (!templatesId.isEmpty())
        {
            GalleryItem galleryItem = new GalleryItem();
            
            GalleryGroup galleryGroup = new GalleryGroup(new I18nizableText("plugin." + _RESOURCES_PLUGIN, "PLUGINS_NEWSLETTER_CATEGORY_TEMPLATESMENU_GROUP_LABEL"));
            galleryItem.addGroup(galleryGroup);
                
            for (String templateId : templatesId)
            {
                String id = this.getId() + "." + templateId;
                
                try
                {
                    NewsletterTemplate template = new NewsletterTemplate(skinId, templateId);
                    LifecycleHelper.setupComponent(template, new SLF4JLoggerAdapter(getLogger()), null, _manager, null, true);
                    template.refreshValues();
                    
                    Configuration conf = _getTemplateItemConfiguration (id, template);
                    galleryItemManager.addComponent(_pluginName, null, id, StaticClientSideElement.class, conf);
                    galleryGroup.addItem(new UnresolvedItem(id, true));
                }
                catch (Exception e)
                {
                    throw new ConfigurationException("Unable to configure local client side element of id " + id, e);
                }
            }
            
            galleryItems.add(galleryItem);
        }
        
        if (galleryItems.size() > 0)
        {
            try
            {
                galleryItemManager.initialize();
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Unable to lookup parameter local components", e);
            }
        }
    }
    
    private boolean _templateExists(String skinId, String id) throws IOException
    {
        TraversableSource skinsDir = (TraversableSource) _sourceResolver.resolveURI("skin:" + skinId + "://newsletter/" + id + "/stylesheets/template.xsl");
        return skinsDir.exists();
    }
    
    /**
     * Get the configuration for a template item
     * @param itemId The item id
     * @param template The newsletter template
     * @return The configuration
     */
    protected Configuration _getTemplateItemConfiguration (String itemId, NewsletterTemplate template)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", itemId);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.ribbon.element.ui.ButtonController");
        
        // enable toggle
        DefaultConfiguration enableToggleConf = new DefaultConfiguration("toggle-enabled");
        enableToggleConf.setValue("true");
        classConf.addChild(enableToggleConf);
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        labelConf.setValue(_i18nUtils.translate(template.getLabel()));
        classConf.addChild(labelConf);
        
        // Description
        DefaultConfiguration descConf = new DefaultConfiguration("description");
        descConf.setValue(_i18nUtils.translate(template.getDescription()));
        classConf.addChild(descConf);
        
        // Name
        DefaultConfiguration nameConf = new DefaultConfiguration("name");
        nameConf.setValue(template.getId());
        classConf.addChild(nameConf);
        
        // Icons
        DefaultConfiguration iconSmallConf = new DefaultConfiguration("icon-small");
        iconSmallConf.setValue(template.getSmallImage());
        classConf.addChild(iconSmallConf);
        DefaultConfiguration iconMediumConf = new DefaultConfiguration("icon-medium");
        iconMediumConf.setValue(template.getMediumImage());
        classConf.addChild(iconMediumConf);
        DefaultConfiguration iconLargeConf = new DefaultConfiguration("icon-large");
        iconLargeConf.setValue(template.getLargeImage());
        classConf.addChild(iconLargeConf);
        
     // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            c.setValue(String.valueOf(commonConfig.get(tagName)));
            classConf.addChild(c);
        }
        
        conf.addChild(classConf);
        return conf;
    }
}
