/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.generators;

import java.io.IOException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Iterator;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.FilenameUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.web.WebConstants;

/**
 * SAX the newsletters of a given category
 *
 */
public class NewsletterListGenerator extends ServiceableGenerator
{
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryProviderEP;

    /** The excalibur source resolver. */
    protected SourceResolver _sourceResolver;
    
    private AmetysObjectResolver _resolver;

    private ContentHelper _contentHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String workspaceName = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);
        
        // Force workspace live for search
        RequestAttributeWorkspaceSelector.setForcedWorkspace(request, WebConstants.LIVE_WORKSPACE);
        
        String siteName = parameters.getParameter("siteName", (String) request.getAttribute(WebConstants.REQUEST_ATTR_SITE_NAME));
        String categoryId = parameters.getParameter("category", null);
        String encodedCategoryId = parameters.getParameter("encodedCategoryId", null);
        boolean saxNewsletterContent = parameters.getParameterAsBoolean("newsletterContent", false);
        String header = parameters.getParameter("header", "");
        int length = parameters.getParameterAsInteger("length", 0);
        boolean displayRss = parameters.getParameterAsBoolean("rss", false);
        
        if (categoryId == null && StringUtils.isNotEmpty(encodedCategoryId))
        {
            categoryId = URIUtils.decode(encodedCategoryId);
        }
        
        Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, "org.ametys.plugins.newsletter.Content.newsletter");
        Expression validExpr = new MetadataExpression(DefaultContent.METADATA_LAST_VALIDATION);
        Expression siteExpr = new StringExpression("site", Operator.EQ, siteName);
        Expression categoryExpr = new StringExpression("category", Operator.EQ, categoryId, ExpressionContext.newInstance().withInternal(true));
        
        Expression expression = new AndExpression(cTypeExpr, validExpr, siteExpr, categoryExpr);
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addCriterion(DefaultContent.METADATA_LAST_VALIDATION, false, false);
        String xpathQuery = org.ametys.plugins.repository.query.QueryHelper.getXPathQuery(null, "ametys:content", expression, sortCriteria);
        
        try (AmetysObjectIterable<Content> contents = _resolver.query(xpathQuery);)
        {
            AttributesImpl attrs = new AttributesImpl();
            if (StringUtils.isNotEmpty(header))
            {
                attrs.addCDATAAttribute("service-title", header);
            }
            
            Category category = categoryId != null ? _categoryProviderEP.getCategory(categoryId) : null;
            
            contentHandler.startDocument();
            XMLUtils.startElement(contentHandler, "Newsletters", attrs);
            
            if (category != null)
            {
                _saxCategory(category);
            }
            
            int index = 0;
            Iterator<Content> it = contents.iterator();
            while (it.hasNext() && index < length)
            {
                Content content = it.next();
                _saxNewsletter(content, saxNewsletterContent);
                index++;
            }
            
            if (displayRss && category != null)
            {
                _saxRssFeed(category);
            }
            
            XMLUtils.endElement(contentHandler, "Newsletters");
            contentHandler.endDocument();
            
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, workspaceName);
        }
    }

    /**
     * Sax a category.
     * @param category the category to sax
     * @throws SAXException if an error occurred while saxing
     */
    protected void _saxCategory(Category category) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", category.getId());
        
        XMLUtils.startElement(contentHandler, "category", attrs);
        
        category.getTitle().toSAX(contentHandler, "title");
        category.getDescription().toSAX(contentHandler, "description");
        
        XMLUtils.endElement(contentHandler, "category");
    }
    
    private void _saxNewsletter (Content content, boolean saxContent) throws SAXException, IOException
    {
        AttributesImpl attr = new AttributesImpl();
        attr.addAttribute("", "id", "id", "CDATA", content.getId());
        attr.addAttribute("", "name", "name", "CDATA", content.getName());
        XMLUtils.startElement(contentHandler, "newsletter", attr);
        
        XMLUtils.createElement(contentHandler, "title", content.getTitle());
        
        long number = content.getValue("newsletter-number", false, 0L);
        if (number != 0)
        {
            XMLUtils.createElement(contentHandler, "number", String.valueOf(number));
        }
        
        LocalDate defaultValue = Optional.ofNullable(content.getLastMajorValidationDate())
                                         .map(ZonedDateTime::toInstant)
                                         .map(DateUtils::asLocalDate)
                                         .orElse(null);
        LocalDate date = content.getValue("newsletter-date", false, defaultValue);
        @SuppressWarnings("unchecked")
        ElementType<LocalDate> dateType = (ElementType<LocalDate>) content.getType("newsletter-date");
        
        AttributesImpl dateAttrs = new AttributesImpl();
        Long millis = date.atStartOfDay(ZoneId.systemDefault()).toInstant().toEpochMilli();
        dateAttrs.addCDATAAttribute("millis", Long.toString(millis));
        XMLUtils.createElement(contentHandler, "date", dateAttrs, dateType.toString(date));
        
        ZonedDateTime pubDate = content.getLastMajorValidationDate();
        dateAttrs.clear();
        dateAttrs.addCDATAAttribute("millis", Long.toString(pubDate.toInstant().toEpochMilli()));
        XMLUtils.createElement(contentHandler, "pubDate", dateAttrs, DateUtils.zonedDateTimeToString(pubDate));
        
        if (saxContent)
        {
            _saxNewsletterContent(content);
        }
        
        XMLUtils.endElement(contentHandler, "newsletter");
    }

    /**
     * Sax the newsletter content.
     * @param content the newsletter content.
     * @throws SAXException if an error occurred while saxing
     * @throws IOException if an I/O error occurred
     */
    protected void _saxNewsletterContent(Content content) throws SAXException, IOException
    {
        XMLUtils.startElement(contentHandler, "newsletter-content");
        
        String uri = _contentHelper.getContentViewUrl(content, "main", "xml");
        SitemapSource src = null;
        
        try
        {
            src = (SitemapSource) _sourceResolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(contentHandler));
        }
        finally
        {
            _sourceResolver.release(src);
        }
        
        XMLUtils.endElement(contentHandler, "newsletter-content");
    }
    
    /**
     * Sax the link to the newsletter archives RSS feed.
     * @param category the category.
     * @throws SAXException if an error occurred while saxing 
     * @throws ProcessingException if an error occurred 
     */
    protected void _saxRssFeed(Category category) throws SAXException, ProcessingException
    {
        String categoryId = category.getId();
        String siteName = category.getSiteName();
        String lang = category.getLang();
        
        String encodedCategoryId = FilenameUtils.encodePath(categoryId);
        String url = "plugins/newsletter/" + siteName + "/" + lang + "/archives/" + encodedCategoryId + "/rss.xml";
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("url", URIUtils.encodePath(url));
        
        XMLUtils.createElement(contentHandler, "rss", attrs);
    }
}
