/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * SAX events for categories stored in JCR repository
 * 
 */
public class NewslettersTreeGenerator extends CategoryGenerator
{
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String categoryID = request.getParameter("categoryID");
        String siteName = request.getParameter("sitename");
        String lang = request.getParameter("lang");
        boolean categoriesOnly = parameters.getParameterAsBoolean("categoriesOnly", false);
        
        contentHandler.startDocument();
        if (StringUtils.isEmpty(categoryID) || "root".equals(categoryID))
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addAttribute("", "id", "id", "CDATA", "root");
            attr.addAttribute("", "hasChild", "hasChild", "CDATA", "true");
            
            XMLUtils.startElement(contentHandler, "provider", attr);
            
            Set<String> ids = _categoryProviderEP.getExtensionsIds();
            for (String id : ids)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(id); 
                saxProvider(provider, siteName, lang, false);
            }
            
            XMLUtils.endElement(contentHandler, "provider");
        }
        else if (categoryID.startsWith("provider_"))
        {
            String id = categoryID.substring("provider_".length());
            CategoryProvider provider = _categoryProviderEP.getExtension(id); 
            
            saxProvider(provider, siteName, lang, true);
        }
        else
        {
            Set<String> ids = _categoryProviderEP.getExtensionsIds();
            for (String id : ids)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(id);
                if (provider.hasCategory(categoryID))
                {
                    Category category = provider.getCategory(categoryID);
                    saxCategory(provider, category, true, !categoriesOnly);
                }
            }
        }
        contentHandler.endDocument();
    }
    
    /**
     * SAX a category provider
     * @param provider the category provider
     * @param siteName the site name
     * @param lang the language name
     * @param children true to SAX children
     * @throws SAXException If the error occurred while SAXing
     */
    protected void saxProvider (CategoryProvider provider, String siteName, String lang, boolean children) throws SAXException
    {
        String id = provider.getId();
        I18nizableText title = provider.getLabel();
        I18nizableText description = provider.getDescription();
        
        AttributesImpl attr = new AttributesImpl();
        attr.addAttribute("", "id", "id", "CDATA", "provider_" + id);
        
        List<Category> childCategories = provider.getCategories(siteName, lang);
        if (childCategories.size() > 0)
        {
            attr.addAttribute("", "hasChild", "hasChild", "CDATA", "true");
        }
        attr.addAttribute("", "mode", "mode", "CDATA", provider.isWritable() ? "write" : "read");
        
        XMLUtils.startElement(contentHandler, "provider", attr);
        
        XMLUtils.startElement(contentHandler, "title");
        title.toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "title");

        XMLUtils.startElement(contentHandler, "description");
        description.toSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "description");
        
        if (children)
        {
            for (Category child : childCategories)
            {
                saxCategory(provider, child, false, false);
            }
        }
        
        XMLUtils.endElement(contentHandler, "provider");
    }
}
