/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
/**
 * This factory creates Ametys.message.MessageTarget for a newsletter category.
 * 
 * See #createTargets to know more.
 * @private
 */
Ext.define("Ametys.plugins.newsletter.NewsletterCategoryMessageTargetFactory", {
	extend: "Ametys.message.factory.DefaultMessageTargetFactory",
	
	/**
	 * Create the targets for a message
	 * @param {Object} parameters The parameters needed by the factory to create the message. Can not be null. Handled elements are
	 * @param {String[]} parameters.ids The categories identifiers. Must be present if categories is empty
	 * @param {Ametys.cms.newsletter.NewsletterCategory[]} parameters.categories The categories themselves. Must be present if ids is empty
	 * @param {Function} callback The callback function called when the targets are created. Parameters are
	 * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
	 */
	createTargets: function(parameters, callback)
	{
		var targets = [];
		
		if (parameters.ids)
		{
			Ametys.cms.newsletter.CategoryDAO.getCategories([parameters.ids], this._createTargets, {scope: this, arguments: {callback: callback, parameters: parameters}});
		}
		else if (parameters.categories)
		{
			this._createTargets (parameters.categories, {callback: callback, parameters: parameters});
		}
	},
	
	/**
	 * @private
	 * Create the category targets
	 * @param {Ametys.cms.newsletter.NewsletterCategory[]} categories The categories
	 * @param {Object} args The arguments
	 * @param {Function} args.callback The callback function called when the targets are created. Parameters are
	 * @param {Ametys.message.MessageTarget[]} args.targets The targets created. Cannot be null.
	 * @param {Object} args.parameters The initial parameters of the #createTargets method
	 */
	_createTargets: function (categories, args)
	{
		delete args.parameters['ids'];
		delete args.parameters['categories'];
		
		var targets = [];
		
		for (var i=0; i < categories.length; i++)
		{
			targets.push(Ext.create("Ametys.message.MessageTarget", {
					id: this.getId(),
					parameters: Ext.merge(categories[i].getProperties(args.parameters || {}), {category: categories[i]})
				})
			);
		}
			
		args.callback(targets);
	}
	
});

Ext.define("Ametys.message.NewsletterMessageTarget", {
	override: "Ametys.message.MessageTarget",
	
	statics:
	{
		/**
		 * @member Ametys.message.MessageTarget
		 * @readonly
		 * @property {String} NEWSLETTER_CATEGORY The target type is a newsletter category. See Ametys.plugins.newsletter.NewsletterCategoryMessageTargetFactory parameters to know more of the associated parameters. 
		 */
		NEWSLETTER_CATEGORY: "newsletter-category",
		
		/**
		 * @member Ametys.message.MessageTarget
		 * @readonly
		 * @property {String} NEWSLETTER_CATEGORY_PROVIDER The target type is a newsletter category provider. The expected parameters are:
		 * @property {String} NEWSLETTER_CATEGORY_PROVIDER.id The id of the provider
		 * @property {String} NEWSLETTER_CATEGORY_PROVIDER.lang The language of the provider
		 * @property {String} NEWSLETTER_CATEGORY_PROVIDER.mode The mode of the provider
		*/
		NEWSLETTER_CATEGORY_PROVIDER: "newsletter-category-provider",
		
		/**
		 * @member Ametys.message.MessageTarget
		 * @readonly
		 * @property {String} NEWSLETTER_SUBSCRIBER The target type is a subscriber to a newsletter. The expected parameters are:
		 * @property {String} NEWSLETTER_SUBSCRIBER.email The email of the subscriber
		*/
		NEWSLETTER_SUBSCRIBER: "newsletter-subscriber"
	}
});
