/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf;

import java.util.List;

import org.apache.avalon.framework.component.Component;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.export.impl.odf.db.LienHelper;
import org.ametys.odf.ose.export.impl.odf.db.ScenarioLienHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Get the queries to export the hierarchy links (LIEN and SCENARIO_LIEN).
 */
public class HierarchyExporter extends AbstractLogEnabled implements Component
{
    /** Avalon Role */
    public static final String ROLE = HierarchyExporter.class.getName();

    /**
     * Get queries to export the link between the parent and the child element.
     * @param parent The parent element
     * @param child The child element
     * @param oseCatalog The OSE catalog
     * @return A {@link List} of {@link ParameterizableQuery} to export the link between the given elements
     */
    public List<ParameterizableQuery> getQueries(ProgramItem parent, ProgramItem child, Long oseCatalog)
    {
        String linkCode = oseCatalog + "_" + parent.getCode() + "-" + child.getCode();
        
        Double weight = 1.0;
        Long minChoice = null;
        Long maxChoice = null;
        if (parent instanceof CourseList)
        {
            CourseList courseList = (CourseList) parent;
            switch (courseList.getType())
            {
                case OPTIONAL:
                    weight = 0.0;
                    break;
                case CHOICE:
                    minChoice = courseList.getMinNumberOfCourses();
                    maxChoice = courseList.getMaxNumberOfCourses();
                    long size = courseList.getCourses().size();
                    if (minChoice > size)
                    {
                        getLogger().warn("[{}] La liste contient min={} mais n'a que {} éléments.", courseList.getId(), minChoice, size);
                    }
                    else
                    {
                        weight = minChoice / (double) size;
                    }
                    break;
                default:
                    break;
            }
        }
        
        return List.of(
            LienHelper.insertInto(linkCode, oseCatalog + "_" + parent.getCode(), oseCatalog + "_" + child.getCode()),
            ScenarioLienHelper.insertInto(linkCode, weight, minChoice, maxChoice)
        );
    }
}
