/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf;

import java.util.Collection;
import java.util.stream.Collectors;

import org.slf4j.Logger;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.runtime.config.Config;

/**
 * Log utils.
 */
public final class LogUtils
{
    private LogUtils()
    {
        // Utils
    }
    
    /**
     * Log as error if it's impossible to export the step because degrees and/or orgunits are not unique or are empty.
     * @param logger The logger
     * @param container The concerned object
     * @param degrees The degrees
     * @param orgUnits The orgunits
     */
    public static void stepImpossibilityDegreesOrOrgunits(Logger logger, Container container, Collection<OdfReferenceTableEntry> degrees, Collection<OrgUnit> orgUnits)
    {
        logger.error("[{}] Impossible to export the step '{}' ({}) to OSE, the degree ({}) or the orgunit ({}) can't be defined.",
            container.getId(),
            container.getTitle(),
            container.getCode(),
            _readableRefTablesEntries(degrees),
            _readableOrgUnits(orgUnits)
        );
    }

    /**
     * Log as error if it's impossible to export the step because there is not associatied functional domain.
     * @param logger The logger
     * @param container The concerned object
     */
    public static void stepImpossibilityFunctDomain(Logger logger, Container container)
    {
        logger.error("[{}] Impossible to export the step '{}' ({}) to OSE, there is no associated functional domain.",
            container.getId(),
            container.getTitle(),
            container.getCode()
        );
    }
    
    /**
     * Log as error if it's impossible to export the program element because steps and/or orgunits are not unique or are empty.
     * @param <T> The type of the program element, should be a subclasse of {@link ProgramItem} and {@link Content}
     * @param logger The logger
     * @param programElement The concerned object
     * @param steps The steps
     * @param orgUnits The orgunits
     */
    public static <T extends ProgramItem & Content> void programElementImpossibilityStepsOrOrgunits(Logger logger, T programElement, Collection<Container> steps, Collection<OrgUnit> orgUnits)
    {
        logger.error("[{}] Impossible to export the program element '{}' ({}) to OSE, the step holder ({}) or the orgunit ({}) can't be defined.",
            programElement.getId(),
            programElement.getTitle(),
            programElement.getCode(),
            _readableProgramElements(steps),
            _readableOrgUnits(orgUnits)
        );
    }

    /**
     * Log as warning if it's impossible to establish the orgunit to export on the program element because there is several values.
     * @param <T> The type of the program element, should be a subclasse of {@link ProgramItem} and {@link Content}
     * @param logger The logger
     * @param programElement The concerned object
     * @param orgUnits The orgunits
     */
    public static <T extends ProgramItem & Content> void programElementWarningOrgUnits(Logger logger, T programElement, Collection<OrgUnit> orgUnits)
    {
        logger.warn("[{}] There are several orgunits ({}) on the program element '{}' ({}), by default the first one has been selected.",
            programElement.getId(),
            _readableOrgUnits(orgUnits),
            programElement.getTitle(),
            programElement.getCode()
        );
    }
    
    /**
     * Log as warning if it's impossible the period type of the program element because there is several values.
     * @param <T> The type of the program element, should be a subclasse of {@link ProgramItem} and {@link Content}
     * @param logger The logger
     * @param programElement The concerned object
     * @param periodTypes The period types
     */
    public static <T extends ProgramItem & Content> void programElementWarningPeriodTypes(Logger logger, T programElement, Collection<OdfReferenceTableEntry> periodTypes)
    {
        logger.warn("[{}] Impossible to export the period type of program element '{}' ({}) to OSE, there are the following values : {}.",
            programElement.getId(),
            programElement.getTitle(),
            programElement.getCode(),
            _readableRefTablesEntries(periodTypes)
        );
    }

    /**
     * Log as debug if it's impossible to export the program element because the step holder is not in the steps of the program element.
     * @param <T> The type of the program element, should be a subclasse of {@link ProgramItem} and {@link Content}
     * @param logger The logger
     * @param programElement The concerned object
     * @param stepHolder The step holder
     * @param steps The steps
     */
    public static <T extends ProgramItem & Content> void programElementDebugIsNotContainedInSteps(Logger logger, T programElement, Container stepHolder, Collection<Container> steps)
    {
        if (logger.isDebugEnabled())
        {
            String readableSteps = _readableProgramElements(steps);
            logger.debug(
                "[{}] The program element '{}' ({}) will not be exported to OSE, as the step holder '{}' ({}) is not among the steps [{}].", 
                programElement.getId(), 
                programElement.getTitle(), 
                programElement.getCode(), 
                stepHolder.getTitle(), 
                stepHolder.getCode(), 
                readableSteps
            );
        }
    }
    
    private static <T extends ProgramItem & Content> String _readableProgramElements(Collection<T> programElements)
    {
        return programElements.stream()
            .map(programElement -> programElement.getTitle() + " (" + programElement.getCode() + ")")
            .collect(Collectors.joining(", "));
    }

    private static String _readableRefTablesEntries(Collection<OdfReferenceTableEntry> entries)
    {
        return entries.stream()
            .map(entry -> entry.getLabel(Config.getInstance().getValue("odf.programs.lang")) + " (" + entry.getCode() + ")")
            .collect(Collectors.joining(", "));
    }
    
    private static String _readableOrgUnits(Collection<OrgUnit> orgUnits)
    {
        return orgUnits.stream()
                .map(orgUnit -> orgUnit.getTitle() + " (" + orgUnit.getUAICode() + ")")
                .collect(Collectors.joining(", "));
    }
}
