/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.db.column;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;

/**
 * The definition of a colum with a name, a SQL type and if it's nullable.
 * Also, we can define a view column name for the associated SRC_* view.
 */
public class DefaultColumn implements Column
{
    /** The name */
    protected String _name;
    /** The SQL type */
    protected String _type;
    /** Column is nullable ? */
    protected boolean _isNullable;
    /** The column name in the view */
    protected String _viewName;

    /**
     * Constructor.
     * @param name The name of the column
     * @param type The SQL type of the column
     * @param isNullable <code>true</code> if it's nullable
     * @param viewName The name of the column in the associated SRC_* view
     */
    public DefaultColumn(String name, String type, boolean isNullable, String viewName)
    {
        this._name = name;
        this._type = type;
        this._isNullable = isNullable;
        this._viewName = viewName;
    }
    
    /**
     * Constructor
     * @param name The name of the column (it will also be used for the associated SRC_* view)
     * @param type The SQL type of the column
     * @param isNullable <code>true</code> if it's nullable
     */
    public DefaultColumn(String name, String type, boolean isNullable)
    {
        this(name, type, isNullable, name);
    }

    @Override
    public String getColumnDescription()
    {
        StringBuilder sb = new StringBuilder(_name);
        sb.append(" ");
        sb.append(_type);
        
        if (!_isNullable)
        {
            sb.append(" NOT NULL");
        }
        
        return sb.toString();
    }

    @Override
    public String getViewDescription()
    {
        if (_viewName == null)
        {
            return StringUtils.EMPTY;
        }
        
        StringBuilder sb = new StringBuilder();
        sb.append("TBL.");
        sb.append(_name);
        sb.append(" AS ");
        sb.append(_viewName);
        
        return sb.toString();
    }

    @Override
    public List<ValuedQueryParameter> getViewParameters()
    {
        return new ArrayList<>();
    }
}
