/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.List;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract class for OSE export.
 */
public abstract class AbstractOSEExport extends AbstractLogEnabled implements OSEExport
{
    @Override
    public void populate(String ametysCatalog, Long oseCatalog)
    {
        _executeQueries(_populate(ametysCatalog, oseCatalog));
    }
    
    /**
     * Populate the Ametys table(s) with the values from Ametys.
     * @param ametysCatalog The Ametys catalog to filter data.
     * @param oseCatalog The OSE catalog to insert into the right year.
     * @return A {@link List} of queries
     */
    protected abstract List<ParameterizableQuery> _populate(String ametysCatalog, Long oseCatalog);
    
    private void _executeQueries(List<ParameterizableQuery> queries)
    {
        String oseDatasource = Config.getInstance().getValue("ose.datasource");
        try (Connection connection = ConnectionHelper.getConnection(oseDatasource))
        {
            for (ParameterizableQuery query : queries)
            {
                _executeQuery(query, connection);
            }
        }
        catch (SQLException e)
        {
            getLogger().error("Error during connection to the database.", e);
        }
    }
    
    private void _executeQuery(ParameterizableQuery query, Connection connection)
    {
        try (PreparedStatement stmt = connection.prepareStatement(query.getQuery()))
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Will execute {}", query.toReadableString());
            }
            _setStatementParameters(stmt, query);
            stmt.execute();
        }
        catch (SQLException e)
        {
            getLogger().error(
                "Impossible to execute the {}",
                query.toReadableString(),
                e
            );
        }
    }
    
    private void _setStatementParameters(PreparedStatement stmt, ParameterizableQuery query) throws SQLException
    {
        int index = 1;
        for (ValuedQueryParameter parameter : query.getParameters())
        {
            stmt.setObject(index, parameter.getValue(), parameter.getType());
            index++;
        }
    }
}
