/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.export.impl.odf.db.CheminPedagogiqueHelper;
import org.ametys.odf.ose.export.impl.odf.db.EffectifsHelper;
import org.ametys.odf.ose.export.impl.odf.db.ElementPedagogiqueHelper;
import org.ametys.odf.ose.export.impl.odf.db.TypeInterventionEPHelper;
import org.ametys.odf.ose.export.impl.odf.db.VolumeHoraireEnsHelper;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.plugins.odfpilotage.cost.entity.Effectives;
import org.ametys.plugins.odfpilotage.cost.entity.Groups;
import org.ametys.plugins.odfpilotage.cost.entity.ProgramItemData;
import org.ametys.runtime.model.ModelHelper;

/**
 * Exporter of courses.
 */
public class CourseExporter extends AbstractProgramElementExporter<Course>
{
    /** Avalon Role */
    public static final String ROLE = CourseExporter.class.getName();
    
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }

    @Override
    protected List<ParameterizableQuery> _getQueries(Course programElement, ProgramElementData data, Long oseCatalog, CostComputationData costData)
    {
        String courseCode = programElement.getCode();
        String courseTitle = _getCourseTitle(programElement);

        /*
         *  ELEMENT_PEDAGOGIQUE
         *  NOEUD
         *  Si ELP de dernier niveau (course.getCourseLists().isEmpty())
         *      Si heures d'enseignement partagées (portées ou non)
         *          Pour chaque heures d'enseignement uniquement portées
         *              ELEMENT_PEDAGOGIQUE
         *              NOEUD
         *              CHEMIN_PEDAGOGIQUE (1 par STEP)
         *              EFFECTIFS
         *              VOLUME_HORAIRE (courseLink = coursePart)
         *              TYPE_INTERVENTION_EP (courseLink = coursePart)
         *      Sinon
         *          CHEMIN_PEDAGOGIQUE (1 par STEP)
         *          EFFECTIFS (Premier CoursePart)
         *          Pour chaque heures d'enseignement uniquement portées
         *              VOLUME_HORAIRE (courseLink = course)
         *              TYPE_INTERVENTION_EP (courseLink = course)
         */
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.addAll(ElementPedagogiqueHelper.insertInto(courseCode, courseTitle, oseCatalog, data.getOrgUnit(), data.getStepHolder(), data.getPeriodType()));

        if (programElement.getCourseLists().isEmpty())
        {
            // Check if one of the course part is shared
            List<CoursePart> courseParts = programElement.getCourseParts();
            boolean hasSharedCoursePart = _hasSharedCoursePart(courseParts);
            CoursePart[] holdedCourseParts = courseParts.stream()
                .filter(coursePart -> _isCourseHolder(coursePart, programElement))
                .toArray(CoursePart[]::new);
            
            if (hasSharedCoursePart)
            {
                for (CoursePart coursePart : holdedCourseParts)
                {
                    String coursePartCode = coursePart.getCode();
                    String coursePartNature = _refTableHelper.getItemCode(coursePart.getNature());
                    String coursePartLabel = courseTitle + " - " + coursePartNature;
                    
                    queries.addAll(ElementPedagogiqueHelper.insertInto(coursePartCode, coursePartLabel, oseCatalog, data.getOrgUnit(), data.getStepHolder(), data.getPeriodType()));
                    
                    queries.addAll(_getCheminPedagogiqueQueries(data.getSteps(), coursePartCode, oseCatalog));
                    
                    _getEffectifsQuery(coursePart, coursePartCode, oseCatalog, costData)
                        .ifPresent(queries::add);
                    
                    queries.addAll(_getCoursePartQueries(coursePart, coursePartNature, oseCatalog, coursePart.getCode(), costData));
                }
            }
            else
            {
                queries.addAll(_getCheminPedagogiqueQueries(data.getSteps(), courseCode, oseCatalog));
                if (!courseParts.isEmpty())
                {
                    _getEffectifsQuery(courseParts.get(0), courseCode, oseCatalog, costData)
                        .ifPresent(queries::add);
                    
                    for (CoursePart coursePart : holdedCourseParts)
                    {
                        String coursePartNature = _refTableHelper.getItemCode(coursePart.getNature());
                        queries.addAll(_getCoursePartQueries(coursePart, coursePartNature, oseCatalog, courseCode, costData));
                    }
                }
            }
        }
        
        return queries;
    }
    
    private Optional<ParameterizableQuery> _getEffectifsQuery(CoursePart coursePart, String code, Long oseCatalog, CostComputationData costData)
    {
        return Optional.ofNullable(coursePart)
            .map(c -> _getNumberOfStudents(c, costData))
            .map(nbStudents -> EffectifsHelper.insertInto(coursePart.getCode(), nbStudents, oseCatalog, code));
    }
    
    private List<ParameterizableQuery> _getCheminPedagogiqueQueries(Set<Container> steps, String code, Long oseCatalog)
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        
        AtomicInteger order = new AtomicInteger();
        for (Container step : steps)
        {
            queries.add(CheminPedagogiqueHelper.insertInto(step.getCode(), code, oseCatalog, order));
        }
        
        return queries;
    }

    private List<ParameterizableQuery> _getCoursePartQueries(CoursePart coursePart, String coursePartNature, Long oseCatalog, String courseLink, CostComputationData costData)
    {
        String coursePartCode = coursePart.getCode();
        Double numberOfGroups = _getGroups(coursePart, costData);
        return List.of(
            VolumeHoraireEnsHelper.insertInto(coursePart, coursePartCode, coursePartNature, oseCatalog, courseLink, numberOfGroups),
            TypeInterventionEPHelper.insertInto(coursePartCode, coursePartNature, courseLink, oseCatalog)
        );
    }

    private boolean _hasSharedCoursePart(List<CoursePart> courseParts)
    {
        return courseParts.stream()
                .map(CoursePart::getCourses)
                .map(List::size)
                .anyMatch(size -> size > 2);
    }
    
    private boolean _isCourseHolder(CoursePart coursePart, Course currentCourse)
    {
        final String currentCourseId = currentCourse.getId();
        return Optional.of(coursePart)
                .map(CoursePart::getCourseHolder)
                .map(Course::getId)
                .map(id -> id.equals(currentCourseId))
                .orElse(false);
    }
    
    private Long _getNumberOfStudents(CoursePart coursePart, CostComputationData costData)
    {
        return Optional.of(coursePart)
                .map(CoursePart::getId)
                .map(costData::get)
                .map(ProgramItemData::getEffectives)
                .flatMap(Effectives::getComputedEffective)
                .map(Double::longValue)
                .orElse(0L);
    }
    
    private Double _getGroups(CoursePart coursePart, CostComputationData costData)
    {
        Optional<Groups> groups = Optional.of(coursePart)
            .map(CoursePart::getId)
            .map(costData::get)
            .map(ProgramItemData::getGroups);
        
        if (groups.isPresent())
        {
            return groups.map(Groups::getGroupsToOpen)
                    .or(() -> groups.map(Groups::getComputedGroups))
                    .map(Long::doubleValue)
                    .orElse(null);
        }
        
        return null;
    }
    
    private String _getCourseTitle(Course course)
    {
        String title = course.getTitle();
        if (ModelHelper.hasModelItem("elpCode", course.getModel()))
        {
            String apogee = course.getValue("elpCode");
            if (StringUtils.isNotEmpty(apogee))
            {
                title += " (" + apogee + ")";
            }
        }
        
        return title;
    }
}

