/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.db;

import java.util.ArrayList;
import java.util.List;

import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;

/**
 * Object representing a query and its parameters.
 */
public class ParameterizableQuery
{
    private String _query;
    private List<ValuedQueryParameter> _parameters;
    
    /**
     * Constructor.
     * @param query The query
     * @param parameters The query parameters
     */
    public ParameterizableQuery(String query, List<ValuedQueryParameter> parameters)
    {
        this._query = query;
        this._parameters = parameters;
    }
    
    /**
     * Constructor.
     * @param query The query
     */
    public ParameterizableQuery(String query)
    {
        this(query, new ArrayList<>());
    }
    
    /**
     * Get the query
     * @return the query
     */
    public String getQuery()
    {
        return _query;
    }
    
    /**
     * Get the query parameters
     * @return the parameters
     */
    public List<ValuedQueryParameter> getParameters()
    {
        return _parameters;
    }
    
    /**
     * Gets the representation of this parameterizable query as a readable string for logging purposes.
     * @return the representation of this parameterizable query as a readable string for logging purposes.
     */
    public String toReadableString()
    {
        StringBuilder sb = new StringBuilder();
        // Set the query
        sb.append("SQL query:");
        sb.append(System.lineSeparator());
        sb.append(getQuery());
        sb.append(System.lineSeparator());
        // Set the parameters
        sb.append("Parameters:");
        for (ValuedQueryParameter param : getParameters())
        {
            sb.append(System.lineSeparator());
            sb.append(" - ");
            sb.append(param.getName());
            sb.append(" = ");
            sb.append(param.getValue());
        }
        return sb.toString();
    }
}
