/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl;

import java.sql.Types;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.cms.data.ContentValue;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.column.DefaultColumn;
import org.ametys.odf.ose.db.column.ForeignKeyColumn;
import org.ametys.odf.ose.db.column.SourceIdColumn;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.AbstractOSEExport;
import org.ametys.odf.ose.export.ExportUtils;
import org.ametys.runtime.config.Config;

/**
 * This exports degrees to GROUPE_TYPE_FORMATION and TYPE_FORMATION tables.
 */
public class TypeFormationExport extends AbstractOSEExport implements Serviceable
{
    private static final String __GROUPE_TYPE_TABLE_NAME = "GROUPE_TYPE_FORMATION";
    private static final String __TYPE_TABLE_NAME = "TYPE_FORMATION";
    
    private static final String __DEGREE_CATEGORY_CONTENT_TYPE = "odf-enumeration.DegreeCategory";
    private static final String __DEGREES_ATTRIBUTE_NAME = "degrees";

    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public List<ParameterizableQuery> initializeDb()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();

        Set<Column> columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new DefaultColumn("LIBELLE_COURT", "VARCHAR2(20 CHAR)", false));
        columns.add(new DefaultColumn("LIBELLE_LONG", "VARCHAR2(50 CHAR)", false));
        columns.add(new DefaultColumn("ORDRE", "NUMBER(*,0)", false));
        columns.add(new SourceIdColumn());
        queries.addAll(ExportUtils.initializeTableAndView(__GROUPE_TYPE_TABLE_NAME, columns));
        
        columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new DefaultColumn("LIBELLE_LONG", "VARCHAR2(80 CHAR)", false));
        columns.add(new DefaultColumn("LIBELLE_COURT", "VARCHAR2(15 CHAR)", false));
        columns.add(new ForeignKeyColumn("GROUPE_ID", "VARCHAR2(100 CHAR)", false, Pair.of("GROUPE_TYPE_FORMATION", "SOURCE_CODE"), false));
        columns.add(new SourceIdColumn());
        queries.addAll(ExportUtils.initializeTableAndView(__TYPE_TABLE_NAME, columns));
        
        return queries;
    }

    @Override
    public List<ParameterizableQuery> _populate(String ametysCatalog, Long oseCatalog)
    {
        List<ParameterizableQuery> queries = new ArrayList<>();

        // Remove all values
        queries.add(ExportUtils.deleteFromAmetys(__TYPE_TABLE_NAME));
        queries.add(ExportUtils.deleteFromAmetys(__GROUPE_TYPE_TABLE_NAME));
        
        // Add all values
        queries.addAll(_getInsertQueries());
        
        return queries;
    }

    private List<ParameterizableQuery> _getInsertQueries()
    {
        String lang = Config.getInstance().getValue("odf.programs.lang");
        
        return _refTableHelper.getItems(__DEGREE_CATEGORY_CONTENT_TYPE)
            .stream()
            .sorted(Comparator
                    .comparingLong(OdfReferenceTableEntry::getOrder)
                    .thenComparing(OdfReferenceTableEntry::getCode))
            .flatMap(degreeCategory -> _getInsertQueries(degreeCategory, lang))
            .collect(Collectors.toList());
    }
    
    private Stream<ParameterizableQuery> _getInsertQueries(OdfReferenceTableEntry degreeCategory, String lang)
    {
        Stream.Builder<ParameterizableQuery> queries = Stream.builder();
        
        // Add the group of types (group of degrees)
        queries.add(_getInsertQueryForGroupType(degreeCategory, lang));
        
        // Add the types (degrees)
        String degreeCategoryCode = degreeCategory.getCode();
        ContentValue[] degrees = degreeCategory.getContent().getValue(__DEGREES_ATTRIBUTE_NAME, false, new ContentValue[0]);
        for (ContentValue degree : degrees)
        {
            degree.getContentIfExists()
                .map(OdfReferenceTableEntry::new)
                .map(entry -> _getInsertQueryForType(entry, lang, degreeCategoryCode))
                .ifPresent(queries::add);
        }

        return queries.build();
    }
    
    private ParameterizableQuery _getInsertQueryForGroupType(OdfReferenceTableEntry degreeCategory, String lang)
    {
        String code = degreeCategory.getCode();
        String label = degreeCategory.getLabel(lang);
        Long order = degreeCategory.getOrder();
        
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE_COURT", StringUtils.truncate(label, 20), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE_LONG", StringUtils.truncate(label, 50), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("ORDRE", order, Types.NUMERIC));
        return ExportUtils.insertIntoAmetys(__GROUPE_TYPE_TABLE_NAME, parameters);
    }
    
    private ParameterizableQuery _getInsertQueryForType(OdfReferenceTableEntry degree, String lang, String degreeCategoryCode)
    {
        String code = degree.getCode();
        String label = degree.getLabel(lang);
        
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE_LONG", StringUtils.truncate(label, 80), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE_COURT", StringUtils.truncate(label, 15), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("GROUPE_ID", degreeCategoryCode, Types.VARCHAR));
        return ExportUtils.insertIntoAmetys(__TYPE_TABLE_NAME, parameters);
    }
    
}
