/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf.db;

import java.sql.Types;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;

import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.column.DefaultColumn;
import org.ametys.odf.ose.db.column.ForeignKeyColumn;
import org.ametys.odf.ose.db.column.SourceIdColumn;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.ExportUtils;

/**
 * Helper to build queries to manage the VOLUME_HORAIRE_ENS table.
 */
public final class VolumeHoraireEnsHelper
{
    private static final String __TABLE_NAME = "VOLUME_HORAIRE_ENS";
    
    private VolumeHoraireEnsHelper()
    {
        // Helper
    }

    /**
     * Queries to create the table and associated view.
     * @return List of queries for table and view creation
     */
    public static List<ParameterizableQuery> initialize()
    {
        Set<Column> columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new DefaultColumn("HEURES", "FLOAT", false));
        columns.add(new ForeignKeyColumn("ELEMENT_PEDAGOGIQUE_ID", "VARCHAR2(100 CHAR)", false, Pair.of("ELEMENT_PEDAGOGIQUE", "SOURCE_CODE"), true));
        columns.add(new ForeignKeyColumn("TYPE_INTERVENTION_ID", "VARCHAR2(10 CHAR)", false, Pair.of("TYPE_INTERVENTION", "CODE"), false));
        columns.add(new DefaultColumn("GROUPES", "FLOAT", true));
        columns.add(new DefaultColumn("ANNEE_ID", "NUMBER(38, 0)", false));
        columns.add(new SourceIdColumn());
        return ExportUtils.initializeTableAndView(__TABLE_NAME, columns);
    }

    /**
     * Query to delete all lines from the table.
     * @return The query for deletion
     */
    public static ParameterizableQuery deleteFrom()
    {
        return ExportUtils.deleteFromAmetys(__TABLE_NAME);
    }

    /**
     * Query to insert a line in the table.
     * @param coursePart The course part
     * @param coursePartCode The course part code
     * @param coursePartNature The course part nature
     * @param oseCatalog The OSE catalog
     * @param courseLink the course link
     * @param numberOfGroups The number of groups
     * @return The query to insert values in the table
     */
    public static ParameterizableQuery insertInto(CoursePart coursePart, String coursePartCode, String coursePartNature, Long oseCatalog, String courseLink, Double numberOfGroups)
    {
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", coursePartCode, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("HEURES", coursePart.getNumberOfHours(), Types.FLOAT));
        parameters.add(new ValuedQueryParameter("ELEMENT_PEDAGOGIQUE_ID", courseLink, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("TYPE_INTERVENTION_ID", coursePartNature, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("GROUPES", numberOfGroups, Types.FLOAT));
        parameters.add(new ValuedQueryParameter("ANNEE_ID", oseCatalog, Types.NUMERIC));
        return ExportUtils.insertIntoAmetys(__TABLE_NAME, parameters);
    }
}
