/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl;

import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.StaticQueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.AbstractOSEExport;
import org.ametys.odf.ose.export.ExportUtils;
import org.ametys.runtime.config.Config;

/**
 * This exports natures like CM, TD, TP to the TYPE_INTERVENTION table.
 * It's a one-shot operation, it's done in the initialization because it's not synchronizable data.
 */
public class NatureEnseignementExport extends AbstractOSEExport implements Serviceable
{
    /** The name for the ETAPE table */
    private static final String __TABLE_NAME = "TYPE_INTERVENTION";
    
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public List<ParameterizableQuery> initializeDb()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.add(_deleteOldData());
        queries.addAll(_getInsertQueries());
        return queries;
    }

    @Override
    protected List<ParameterizableQuery> _populate(String ametysCatalog, Long oseCatalog)
    {
        // Nothing to do, only on initialization, data not synchronizable
        return Collections.EMPTY_LIST;
    }
    
    private Long _orderFromCategory(OdfReferenceTableEntry referenceTableEntry)
    {
        return Optional.of(referenceTableEntry)
                .map(OdfReferenceTableEntry::getContent)
                .map(c -> c.<ContentValue>getValue("category"))
                .flatMap(ContentValue::getContentIfExists)
                .map(c -> c.getValue("order", false, Long.MAX_VALUE))
                .orElse(Long.MAX_VALUE);
    }
    
    private ParameterizableQuery _deleteOldData()
    {
        return ExportUtils.deleteFrom(__TABLE_NAME);
    }
    
    private List<ParameterizableQuery> _getInsertQueries()
    {
        AtomicInteger order = new AtomicInteger();
        String lang = Config.getInstance().getValue("odf.programs.lang");
        return _refTableHelper.getItems(OdfReferenceTableHelper.ENSEIGNEMENT_NATURE)
            .stream()
            .sequential()
            .sorted(Comparator
                .comparingLong(this::_orderFromCategory)
                .thenComparingLong(OdfReferenceTableEntry::getOrder)
                .thenComparing(OdfReferenceTableEntry::getCode))
            .map(enseignementNature -> _getInsertQuery(enseignementNature, order, lang))
            .collect(Collectors.toList());
    }
    
    private ParameterizableQuery _getInsertQuery(OdfReferenceTableEntry enseignementNature, AtomicInteger order, String lang)
    {
        String label = enseignementNature.getLabel(lang);
        
        List<QueryParameter> queryParameters = Arrays.asList(
                new StaticQueryParameter("ID", "TYPE_INTERVENTION_ID_SEQ.NEXTVAL"),
                new ValuedQueryParameter("CODE", enseignementNature.getCode(), Types.VARCHAR),
                new ValuedQueryParameter("LIBELLE", StringUtils.truncate(label, 60), Types.VARCHAR),
                new ValuedQueryParameter("ORDRE", order.incrementAndGet(), Types.NUMERIC),
                new StaticQueryParameter("HISTO_CREATEUR_ID", String.valueOf(1)),
                new StaticQueryParameter("HISTO_MODIFICATEUR_ID", String.valueOf(1)),
                new StaticQueryParameter("VISIBLE", String.valueOf(1))
        );
        
        return ExportUtils.insertInto(__TABLE_NAME, queryParameters);
    }
}
