/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl;

import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.StaticQueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.AbstractOSEExport;
import org.ametys.odf.ose.export.ExportUtils;
import org.ametys.runtime.config.Config;

/**
 * This exports period types like pair, impair to the PERIODE table.
 * It's a one-shot operation, it's done in the initialization because it's not synchronizable data.
 */
public class TypePeriodeExport extends AbstractOSEExport implements Serviceable
{
    /** The name for the ETAPE table */
    private static final String __TABLE_NAME = "PERIODE";
    
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public List<ParameterizableQuery> initializeDb()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.add(_deleteOldData());
        queries.addAll(_getInsertQueries());
        return queries;
    }

    @Override
    protected List<ParameterizableQuery> _populate(String ametysCatalog, Long oseCatalog)
    {
        // Nothing to do, only on initialization, data not synchronizable
        return Collections.EMPTY_LIST;
    }
    
    private ParameterizableQuery _deleteOldData()
    {
        return ExportUtils.deleteFrom(__TABLE_NAME);
    }
    
    private List<ParameterizableQuery> _getInsertQueries()
    {
        AtomicInteger order = new AtomicInteger();
        String lang = Config.getInstance().getValue("odf.programs.lang");
        return _refTableHelper.getItems(OdfReferenceTableHelper.PERIOD_TYPE)
            .stream()
            .sequential()
            .sorted(Comparator
                .comparingLong(OdfReferenceTableEntry::getOrder)
                .thenComparing(OdfReferenceTableEntry::getCode))
            .map(enseignementNature -> _getInsertQuery(enseignementNature, order, lang))
            .collect(Collectors.toList());
    }
    
    private ParameterizableQuery _getInsertQuery(OdfReferenceTableEntry typePeriode, AtomicInteger order, String lang)
    {
        String label = typePeriode.getLabel(lang);
        
        List<QueryParameter> queryParameters = Arrays.asList(
                new StaticQueryParameter("ID", "PERIODE_ID_SEQ.NEXTVAL"),
                new ValuedQueryParameter("CODE", StringUtils.truncate(typePeriode.getCode(), 3), Types.VARCHAR),
                new ValuedQueryParameter("LIBELLE_LONG", StringUtils.truncate(label, 40), Types.VARCHAR),
                new ValuedQueryParameter("LIBELLE_COURT", StringUtils.truncate(label, 15), Types.VARCHAR),
                new ValuedQueryParameter("ORDRE", order.incrementAndGet(), Types.NUMERIC),
                new StaticQueryParameter("ENSEIGNEMENT", String.valueOf(1)),
                new StaticQueryParameter("ECART_MOIS", String.valueOf(0)),
                new StaticQueryParameter("ECART_MOIS_PAIEMENT", String.valueOf(0)),
                new StaticQueryParameter("HISTO_CREATEUR_ID", String.valueOf(1)),
                new StaticQueryParameter("HISTO_MODIFICATEUR_ID", String.valueOf(1))
        );
        
        return ExportUtils.insertInto(__TABLE_NAME, queryParameters);
    }
}
