/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.generators;

import java.io.IOException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.ListUtils;
import org.xml.sax.SAXException;

import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.OSEExport;
import org.ametys.odf.ose.export.OSEExportExtensionPoint;
import org.ametys.runtime.config.Config;

/**
 * The generator which generates the SQL initialization file for Ametys connector.
 */
public class ExportInitGenerator extends ServiceableGenerator
{
    /** The OSE export extension point */
    protected OSEExportExtensionPoint _oseExportEP;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _oseExportEP = (OSEExportExtensionPoint) manager.lookup(OSEExportExtensionPoint.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        List<ParameterizableQuery> queries = _retrieveQueries();
        _saxQueries(queries);
    }
    
    /**
     * Retrieve the queries to SAX
     * @return the queries to SAX
     */
    protected List<ParameterizableQuery> _retrieveQueries()
    {
        return ListUtils.union(
                _retrieveStaticQueries(), 
                _retrieveQueriesFromExtensions()
        );
    }
    
    /**
     * Retrieve the static queries (hardcoded) to SAX
     * @return the static queries (hardcoded) to SAX
     */
    protected List<ParameterizableQuery> _retrieveStaticQueries()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();

        String oseSourceCode = Config.getInstance().getValue("ose.source.code");
        List<ValuedQueryParameter> queryParameters = Collections.singletonList(new ValuedQueryParameter("SOURCE_CODE", oseSourceCode, Types.VARCHAR));
        queries.add(new ParameterizableQuery("DELETE FROM SOURCE WHERE CODE = :SOURCE_CODE", queryParameters));
        queries.add(new ParameterizableQuery("INSERT INTO SOURCE (ID, CODE, LIBELLE, IMPORTABLE) VALUES (SOURCE_ID_SEQ.NEXTVAL, :SOURCE_CODE, 'Ametys', 1)", queryParameters));
        
        return queries;
    }
    
    /**
     * Retrieve the queries from {@link OSEExportExtensionPoint} to SAX
     * @return the queries from {@link OSEExportExtensionPoint} to SAX
     */
    protected List<ParameterizableQuery> _retrieveQueriesFromExtensions()
    {
        return _oseExportEP.getExtensionsIds()
                .stream()
                .map(_oseExportEP::getExtension)
                .map(OSEExport::initializeDb)
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }
    
    /**
     * SAX queries
     * @param queries The queries to SAX
     * @throws SAXException Any SAX exception
     */
    protected void _saxQueries(List<ParameterizableQuery> queries) throws SAXException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "queries");
        
        for (ParameterizableQuery query : queries)
        {
            String queryText = _queryText(query);
            XMLUtils.createElement(contentHandler, "query", queryText);
        }
        
        XMLUtils.endElement(contentHandler, "queries");
        contentHandler.endDocument();
    }
    
    /**
     * Gets the text representation of the query to SAX
     * @param query The query to SAX
     * @return The string representation of the query
     */
    protected String _queryText(ParameterizableQuery query)
    {
        String queryText = query.getQuery();
        for (ValuedQueryParameter parameter : query.getParameters())
        {
            String target = ":" + parameter.getName();
            String replacement = "'" + parameter.getValue().toString().replace("'", "''") + "'";
            queryText = queryText.replace(target, replacement);
        }
        return queryText;
    }

}
