/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.export.utils.ElementRetriever;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Exporter for program elements.
 * @param <T> The type of the element to export, should extend {@link Content} and {@link ProgramItem}
 */
public abstract class AbstractProgramElementExporter<T extends Content & ProgramItem> extends AbstractLogEnabled implements Component, Serviceable
{
    /** The retriever of elements from ODF */
    protected ElementRetriever _elementRetriever;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _elementRetriever = (ElementRetriever) manager.lookup(ElementRetriever.ROLE);
    }
    
    /**
     * Get queries to export the given program element.
     * @param programElement The program element to export
     * @param oseCatalog The OSE catalog
     * @param costData the result of cost computation of the catalog
     * @return A {@link List} of {@link ParameterizableQuery} to export the program element, it can be empty if there is a problem (see logs)
     */
    public List<ParameterizableQuery> getQueries(T programElement, Long oseCatalog, CostComputationData costData)
    {
        // Structure : Remonter parent par parent pour obtenir le ou les structures rattachées (sur abstractProgram)
        Set<OrgUnit> orgUnits = _elementRetriever.retrieveOrgUnits(programElement);
        
        // Get the step holder (étape porteuse en français dans le texte)
        Set<Container> stepsHolder = _elementRetriever.retrieveStepsHolder(programElement);

        // Get the period type on the current program element and parents
        Set<OdfReferenceTableEntry> periodTypes = _elementRetriever.retrievePeriodTypes(programElement);
        
        if (stepsHolder.size() != 1 || orgUnits.size() != 1)
        {
            LogUtils.programElementImpossibilityStepsOrOrgunits(getLogger(), programElement, stepsHolder, orgUnits);
            return List.of();
        }
        
        Container stepHolder = _getFirstStepHolder(stepsHolder);
        Set<Container> steps = _elementRetriever.getSteps(programElement);
        // step holder is not among steps => do not export the program element
        if (!steps.contains(stepHolder))
        {
            LogUtils.programElementDebugIsNotContainedInSteps(getLogger(), programElement, stepHolder, steps);
            return List.of();
        }
        
        ProgramElementData data = new ProgramElementData();
        Optional.of(periodTypes)
            .map(this::_getFirstRefTableCode)
            .ifPresentOrElse(
                data::setPeriodType,
                () -> LogUtils.programElementWarningPeriodTypes(getLogger(), programElement, periodTypes)
            );
        data.setStepHolder(stepHolder.getCode());
        data.setOrgUnit(_getFirstOrgUnitCode(orgUnits));
        data.setSteps(steps);
        
        return _getQueries(programElement, data, oseCatalog, costData);
    }
    
    /**
     * Get the queries to create the program element in the OSE database.
     * @param programElement The program element
     * @param data The calculated data attached to the program element
     * @param oseCatalog The OSE catalog
     * @param costData the result of cost computation of the catalog
     * @return The list of queries
     */
    protected abstract List<ParameterizableQuery> _getQueries(T programElement, ProgramElementData data, Long oseCatalog, CostComputationData costData);
    
    private Container _getFirstStepHolder(Set<Container> stepsHolder)
    {
        return stepsHolder
            .stream()
            .findFirst()
            .get();
    }
    
    private String _getFirstOrgUnitCode(Set<OrgUnit> orgUnits)
    {
        return orgUnits
            .stream()
            .findFirst()
            .map(OrgUnit::getUAICode)
            .get();
    }
    
    private String _getFirstRefTableCode(Set<OdfReferenceTableEntry> entries)
    {
        return entries
            .stream()
            .findFirst()
            .map(OdfReferenceTableEntry::getCode)
            .orElse(null);
    }

    /**
     * An object to represent common computed data on the program element like orgunit, step holder, etc.
     */
    protected static class ProgramElementData
    {
        private String _orgUnitCode;
        private String _stepHolderCode;
        private Set<Container> _steps;
        private String _periodTypeCode;
        
        /**
         * Set the orgunit code of the program element.
         * @param orgUnitCode The code of the orgunit
         */
        public void setOrgUnit(String orgUnitCode)
        {
            _orgUnitCode = orgUnitCode;
        }

        /**
         * Set the step holder code of the program element.
         * @param stepHolderCode The code of the step holder
         */
        public void setStepHolder(String stepHolderCode)
        {
            _stepHolderCode = stepHolderCode;
        }

        /**
         * Set the period type code of the program element.
         * @param periodTypeCode The code of the period type
         */
        public void setPeriodType(String periodTypeCode)
        {
            _periodTypeCode = periodTypeCode;
        }

        /**
         * Set the steps attached to the program element.
         * @param steps The steps
         */
        public void setSteps(Set<Container> steps)
        {
            _steps = steps;
        }

        /**
         * Get the orgunit code of the program element.
         * @return The code of the orgunit
         */
        public String getOrgUnit()
        {
            return _orgUnitCode;
        }

        /**
         * Get the step holder code of the program element.
         * @return The code of the step holder
         */
        public String getStepHolder()
        {
            return _stepHolderCode;
        }

        /**
         * Get the period type code of the program element.
         * @return The code of the period type
         */
        public String getPeriodType()
        {
            return _periodTypeCode;
        }

        /**
         * Get the steps attached to the program element.
         * @return The steps
         */
        public Set<Container> getSteps()
        {
            return _steps;
        }
    }
}
