/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf.db;

import java.sql.Types;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.column.DefaultColumn;
import org.ametys.odf.ose.db.column.ForeignKeyColumn;
import org.ametys.odf.ose.db.column.SourceIdColumn;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.ExportUtils;

/**
 * Helper to build queries to manage the ELEMENT_PEDAGOGIQUE table.
 */
public final class ElementPedagogiqueHelper
{
    private static final String __TABLE_NAME = "ELEMENT_PEDAGOGIQUE";
    
    private ElementPedagogiqueHelper()
    {
        // Helper
    }

    /**
     * Queries to create the table and associated view.
     * @return List of queries for table and view creation
     */
    public static List<ParameterizableQuery> initialize()
    {
        Set<Column> columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new DefaultColumn("LIBELLE", "VARCHAR2(200 CHAR)", false));
        columns.add(new DefaultColumn("CODE", "VARCHAR2(50 CHAR)", false));
        columns.add(new ForeignKeyColumn("ETAPE_ID", "VARCHAR2(100 CHAR)", false, Pair.of("ETAPE", "SOURCE_CODE"), true));
        columns.add(new ForeignKeyColumn("STRUCTURE_ID", "VARCHAR2(100 CHAR)", false, Pair.of("STRUCTURE", "CODE"), false));
        columns.add(new ForeignKeyColumn("PERIODE_ID", "VARCHAR2(3 CHAR)", true, Pair.of("PERIODE", "CODE"), false));
        columns.add(new DefaultColumn("ANNEE_ID", "NUMBER(38, 0)", false));
        columns.add(new SourceIdColumn());
        return ExportUtils.initializeTableAndView(__TABLE_NAME, columns);
    }

    /**
     * Query to delete all lines from the table.
     * @return The query for deletion
     */
    public static ParameterizableQuery deleteFrom()
    {
        return ExportUtils.deleteFromAmetys(__TABLE_NAME);
    }

    private static ParameterizableQuery _insertInto(String code, String label, Long oseCatalog, String orgUnit, String stepHolder, String periodType)
    {
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE", StringUtils.truncate(label, 200), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("CODE", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("STRUCTURE_ID", orgUnit, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("ETAPE_ID", stepHolder, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("PERIODE_ID", StringUtils.truncate(periodType, 3), Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("ANNEE_ID", oseCatalog, Types.NUMERIC));
        return ExportUtils.insertIntoAmetys(__TABLE_NAME, parameters);
    }

    /**
     * Queries to insert a line in the table and in NOEUD and SCENARIO_NOEUD.
     * @param code The course code
     * @param label The label
     * @param oseCatalog The OSE catalog
     * @param orgUnit The orgunit
     * @param stepHolder The step holder
     * @param periodType The type of period
     * @return The list of queries to insert values in the tables
     */
    public static List<ParameterizableQuery> insertInto(String code, String label, Long oseCatalog, String orgUnit, String stepHolder, String periodType)
    {
        String nodeId = oseCatalog + "_" + code;
        
        return List.of(
            _insertInto(code, label, oseCatalog, orgUnit, stepHolder, periodType),
            NoeudHelper.insertInto(nodeId, code, false, label, oseCatalog),
            ScenarioNoeudHelper.insertInto(nodeId)
        );
    }
}
