/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf.db;

import java.sql.Types;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.column.DefaultColumn;
import org.ametys.odf.ose.db.column.ForeignKeyColumn;
import org.ametys.odf.ose.db.column.SourceIdColumn;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.StaticQueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.ExportUtils;

/**
 * Helper to build queries to manage the NOEUD table.
 */
public final class NoeudHelper
{
    private static final String __TABLE_NAME = "NOEUD";
    
    private NoeudHelper()
    {
        // Helper
    }

    /**
     * Queries to create the table and associated view.
     * @return List of queries for table and view creation
     */
    public static List<ParameterizableQuery> initialize()
    {
        Set<Column> columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new DefaultColumn("CODE", "VARCHAR2(50 CHAR)", false));
        columns.add(new DefaultColumn("LIBELLE", "VARCHAR2(255 CHAR)", false));
        columns.add(new DefaultColumn("LISTE", "NUMBER(1)", false));
        columns.add(new ForeignKeyColumn("ETAPE_ID", "VARCHAR2(100 CHAR)", true, Pair.of("ETAPE", "SOURCE_CODE"), true));
        columns.add(new ForeignKeyColumn("ELEMENT_PEDAGOGIQUE_ID", "VARCHAR2(100 CHAR)", true, Pair.of("ELEMENT_PEDAGOGIQUE", "SOURCE_CODE"), true));
        columns.add(new DefaultColumn("ANNEE_ID", "NUMBER(38, 0)", false));
        columns.add(new SourceIdColumn());
        return ExportUtils.initializeTableAndView(__TABLE_NAME, columns);
    }

    /**
     * Query to delete all lines from the table.
     * @return The query for deletion
     */
    public static ParameterizableQuery deleteFrom()
    {
        return ExportUtils.deleteFromAmetys(__TABLE_NAME);
    }

    /**
     * Query to insert a line in the table.
     * @param nodeId The identifier (concatenation of the catalog and the code)
     * @param code The code
     * @param isEtape <code>true</code> if it is a step, otherwise set it to <code>false</code>
     * @param title The title
     * @param oseCatalog The OSE catalog
     * @return The query to insert values in the table
     */
    public static ParameterizableQuery insertInto(String nodeId, String code, boolean isEtape, String title, Long oseCatalog)
    {
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", nodeId, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("CODE", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIBELLE", StringUtils.truncate(title, 255), Types.VARCHAR));
        parameters.add(new StaticQueryParameter("LISTE", String.valueOf(0)));
        parameters.add(new ValuedQueryParameter("ETAPE_ID", isEtape ? code : null, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("ELEMENT_PEDAGOGIQUE_ID", isEtape ? null : code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("ANNEE_ID", oseCatalog, Types.NUMERIC));
        return ExportUtils.insertIntoAmetys(__TABLE_NAME, parameters);
    }
}
