/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.ose.export.OSEConstants;
import org.ametys.odf.ose.export.impl.odf.LogUtils;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.model.ModelHelper;

/**
 * A retriever used like a helper to retrieve some ODF elements from ODF items.
 */
public class ElementRetriever extends PilotageHelper
{
    /** Avalon Role */
    public static final String ROLE = ElementRetriever.class.getName();
    
    /** The cache id for step holders by program item */
    private static final String __STEP_HOLDERS_BY_ITEM_CACHE_ID = ElementRetriever.class.getName() + "$stepHoldersByItem";
    
    /** The resolver */
    protected AmetysObjectResolver _resolver;

    /** The helper for reference tables from ODF */
    protected OdfReferenceTableHelper _refTableHelper; 
     
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }

    @Override
    protected String _getStepsHolderByItemCacheId()
    {
        // The cache id is different than its superclass
        return __STEP_HOLDERS_BY_ITEM_CACHE_ID;
    }
    
    @Override
    protected Set<Container> _getStepsToCache(ProgramItem programItem, String yearId)
    {
        return _mustNotExport(programItem) ? Collections.EMPTY_SET : super._getStepsToCache(programItem, yearId);
    }
    
    private boolean _mustNotExport(ProgramItem programItem)
    {
        if (programItem instanceof Course || programItem instanceof Container)
        {
            return ((Content) programItem).getValue(OSEConstants.NO_OSE_EXPORT_ATTRIBUTE_NAME, false, false);
        }
        else
        {
            return false;
        }
    }
    
    /**
     * Retrieve the {@link OrgUnit}s of a given {@link ProgramItem}
     * @param programItem The program item
     * @return the {@link OrgUnit}s of the given program item
     */
    public Set<OrgUnit> retrieveOrgUnits(ProgramItem programItem)
    {
        return _retrieveOrgUnits(programItem);
    }
    
    private Set<OrgUnit> _retrieveOrgUnits(ProgramItem parent)
    {
        return Optional.of(parent)
                .map(this::_getDirectOrgUnit)
                .map(Collections::singleton)
                .orElseGet(() -> _retrieveOrgUnitsFromParents(parent));
    }
    
    private Set<OrgUnit> _retrieveOrgUnitsFromParents(ProgramItem programItem)
    {
        return _odfHelper.getParentProgramItems(programItem)
                .stream()
                .map(this::_retrieveOrgUnits)
                .flatMap(Set::stream)
                .collect(Collectors.toSet());
    }
    
    private OrgUnit _getDirectOrgUnit(ProgramItem programItem)
    {
        List<String> orgUnitIds = programItem.getOrgUnits();
        return _resolveAllAndGetFirst(orgUnitIds, (ProgramItem & Content) programItem);
    }
    
    private <T extends ProgramItem & Content> OrgUnit _resolveAllAndGetFirst(Collection<String> orgUnitIds, T programElement)
    {
        List<OrgUnit> orgUnits = orgUnitIds.stream()
                .map(this::_resolve)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        
        if (orgUnits.size() > 1)
        {
            LogUtils.programElementWarningOrgUnits(getLogger(), programElement, orgUnits);
        }
        
        return orgUnits.isEmpty() ? null : orgUnits.get(0);
    }
    
    private OrgUnit _resolve(String id)
    {
        try
        {
            return _resolver.resolveById(id);
        }
        catch (AmetysRepositoryException e)
        {
            // Silently fail
            return null;
        }
    }

    /**
     * Retrieve the degrees of a given {@link Container}
     * @param container The container
     * @return the degrees of the given container
     */
    public Set<OdfReferenceTableEntry> retrieveDegree(Container container)
    {
        return _odfHelper.getParentPrograms(container)
                .stream()
                .map(p -> p.<ContentValue>getValue(AbstractProgram.DEGREE))
                .filter(Objects::nonNull)
                .map(ContentValue::getContentIfExists)
                .flatMap(Optional::stream)
                .map(OdfReferenceTableEntry::new)
                .collect(Collectors.toSet());
    }

    /**
     * Get the potential steps holder of the program item. A step is a container of year type and it can be set manually on intermediate courses.
     * @param programItem The {@link ProgramItem} on which we have to retrieve the potential steps holder
     * @return the potential steps holder of the program item
     */
    public Set<Container> retrieveStepsHolder(ProgramItem programItem)
    {
        Optional<String> yearId = _odfHelper.getYearId();
        if (yearId.isPresent())
        {
            return _getStepsHolder(programItem, yearId.get());
        }
        
        return Set.of();
    }
    
    /**
     * Get the potential period types of the program item. It can be retrieved on courses or containers. The algorithm doesn't search in the parent of semester containers.
     * @param programItem The {@link ProgramItem} on which we have to retrieve the potential period types
     * @return the potential period types of the program item
     */
    public Set<OdfReferenceTableEntry> retrievePeriodTypes(ProgramItem programItem)
    {
        return getSemesterId()
            .map(id -> _retrievePeriodTypes(programItem, id))
            .orElse(Collections.EMPTY_SET);
    }

    private Set<OdfReferenceTableEntry> _retrievePeriodTypes(ProgramItem programItem, String semesterId)
    {
        return _getPeriodType(programItem) // Try to get the period type on the current program item
            .map(Collections::singleton)
            .orElseGet(() -> _searchPeriodTypesInParents(programItem, semesterId)); // Or search in the parent program items
    }
    
    private Set<OdfReferenceTableEntry> _searchPeriodTypesInParents(ProgramItem programItem, String semesterId)
    {
        if (_odfHelper.isContainerOfNature(programItem, semesterId))
        {
            return Collections.EMPTY_SET;
        }
        
        // Only if programItem not a semester
        return _odfHelper.getParentProgramItems(programItem) // Get the direct parent program items
            .stream()
            .map(parent -> _retrievePeriodTypes(parent, semesterId)) // Retrieve the period types for all parents
            .flatMap(Set::stream)
            .collect(Collectors.toSet());
    }
    
    private Optional<OdfReferenceTableEntry> _getPeriodType(ProgramItem programItem)
    {
        return Optional.of(programItem)
            .map(Content.class::cast) // Cast to Content (ProgramItem is always a Content)
            .filter(c -> ModelHelper.hasModelItem("period", c.getModel())) // Filter if the attribute "period" is not in the model
            .map(c -> c.<ContentValue>getValue("period")) // Get the period
            .flatMap(ContentValue::getContentIfExists)
            .map(c -> c.<ContentValue>getValue("type")) // Get the period type
            .flatMap(ContentValue::getContentIfExists)
            .map(OdfReferenceTableEntry::new);
    }

    
    /**
     * Get the semester container nature identifier.
     * @return an {@link Optional} of the semester identifier
     */
    public Optional<String> getSemesterId()
    {
        return Optional.of(_refTableHelper)
                .map(rth -> rth.getItemFromCode(OdfReferenceTableHelper.CONTAINER_NATURE, "semestre"))
                .map(OdfReferenceTableEntry::getId)
                .filter(StringUtils::isNotBlank);
    }
}
