/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;

/**
 * Utils method for export
 */
public final class ExportUtils
{
    private static final String __TABLE_PREFIX = "AMETYS_";
    private static final String __VIEW_PREFIX = "SRC_";
    
    private ExportUtils() 
    {
        // Nothing to do
    }

    /**
     * Build a request to insert values into the Ametys table. The baseName will be prefixed by "AMETYS_".
     * @param baseName The name of the OSE table
     * @param parameters The list of the parameters with their name and type.
     * @return A query
     */
    public static ParameterizableQuery insertIntoAmetys(String baseName, List<QueryParameter> parameters)
    {
        return insertInto(__TABLE_PREFIX + baseName, parameters);
    }
    
    /**
     * Build a request to insert values into the table.
     * @param tableName The name of the OSE table
     * @param parameters The list of the parameters with their name and type.
     * @return A query
     */
    public static ParameterizableQuery insertInto(String tableName, List<QueryParameter> parameters)
    {
        StringBuilder sb = new StringBuilder("INSERT INTO ");
        sb.append(tableName);
        sb.append(" (");
        sb.append(
            parameters.stream()
                .sequential()
                .map(QueryParameter::getName)
                .collect(Collectors.joining(", "))
        );
        sb.append(") VALUES (");
        sb.append(
            parameters.stream()
                .sequential()
                .map(QueryParameter::getParameter)
                .collect(Collectors.joining(", "))
        );
        sb.append(")");
        
        return new ParameterizableQuery(
                sb.toString(),
                parameters.stream()
                    .sequential()
                    .filter(ValuedQueryParameter.class::isInstance)
                    .map(ValuedQueryParameter.class::cast)
                    .collect(Collectors.toList())
        );
    }
    
    /**
     * Build the request to delete all the rows from an Ametys table. The baseName will be prefixed by "AMETYS_".
     * @param baseName The name of the OSE table
     * @return A query
     */
    public static ParameterizableQuery deleteFromAmetys(String baseName)
    {
        return deleteFrom(__TABLE_PREFIX + baseName);
    }
    
    /**
     * Build the request to delete all the rows from a table.
     * @param tableName The name of the OSE table
     * @return A query
     */
    public static ParameterizableQuery deleteFrom(String tableName)
    {
        StringBuilder sb = new StringBuilder("DELETE FROM ");
        sb.append(tableName);
        return new ParameterizableQuery(sb.toString());
    }
    
    /**
     * Build requests to drop, create and activate import on the OSE table defined by the parameter baseName.
     * @param baseName The name of the OSE table
     * @param tableColumns The columns for the Ametys table and the source view
     * @return A {@link List} of queries
     */
    public static List<ParameterizableQuery> initializeTableAndView(String baseName, Set<Column> tableColumns)
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.add(_dropView(baseName));
        queries.add(_dropTable(baseName));
        queries.add(_createTable(baseName, tableColumns));
        queries.add(_createView(baseName, tableColumns));
        queries.add(_activateImport(baseName));
        return queries;
    }
    
    private static ParameterizableQuery _dropTable(String baseName)
    {
        StringBuilder sb = new StringBuilder("DROP TABLE ");
        sb.append(__TABLE_PREFIX);
        sb.append(baseName);
        return new ParameterizableQuery(sb.toString());
    }
    
    private static ParameterizableQuery _dropView(String baseName)
    {
        StringBuilder sb = new StringBuilder("DROP VIEW ");
        sb.append(__VIEW_PREFIX);
        sb.append(baseName);
        return new ParameterizableQuery(sb.toString());
    }
    
    private static ParameterizableQuery _createTable(String baseName, Set<Column> tableColumns)
    {
        StringBuilder sb = new StringBuilder("CREATE TABLE ");
        sb.append(__TABLE_PREFIX);
        sb.append(baseName);
        sb.append(" (");
        sb.append(tableColumns.stream()
            .map(column -> column.getColumnDescription())
            .filter(StringUtils::isNotEmpty)
            .collect(Collectors.joining(", ")));
        sb.append(")");
        return new ParameterizableQuery(sb.toString());
    }
    
    private static ParameterizableQuery _createView(String baseName, Set<Column> tableColumns)
    {
        StringBuilder sb = new StringBuilder("CREATE VIEW ");
        sb.append(__VIEW_PREFIX);
        sb.append(baseName);
        sb.append(" AS SELECT ");
        sb.append(tableColumns.stream()
            .map(column -> column.getViewDescription())
            .filter(StringUtils::isNotEmpty)
            .collect(Collectors.joining(", ")));
        sb.append(" FROM ");
        sb.append(__TABLE_PREFIX);
        sb.append(baseName);
        sb.append(" TBL");
        
        return new ParameterizableQuery(
            sb.toString(),
            tableColumns.stream()
                .map(Column::getViewParameters)
                .flatMap(List::stream)
                .collect(Collectors.toList())
        );
    }
    
    private static ParameterizableQuery _activateImport(String baseName)
    {
        StringBuilder sb = new StringBuilder("UPDATE IMPORT_TABLES SET SYNC_ENABLED = 1, SYNC_JOB = 'synchro-ametys', SYNC_FILTRE = NULL WHERE TABLE_NAME = '");
        sb.append(baseName);
        sb.append("'");
        return new ParameterizableQuery(sb.toString());
    }
}

