/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.enumeration;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.JcrConstants;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Global validator for a Degree Category (entry of reference table),
 * which checks that all its degrees are only binded to this category.
 */
public class DegreesInCategoryConsistencyValidator extends AbstractContentValidator implements Serviceable
{
    private static final String __DEGREES_CATEGORY_CONTENT_TYPE = "odf-enumeration.DegreeCategory";
    private static final String __DEGREES_ATTRIBUTE_NAME = "degrees";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public ValidationResult validate(Content content)
    {
        // Nothing to do
        return ValidationResult.empty();
    }
    
    public ValidationResult validate(Content degreeCategory, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        Optional.ofNullable(values.get(__DEGREES_ATTRIBUTE_NAME))
                .map(v -> DataHolderHelper.getValueFromSynchronizableValue(v, degreeCategory, degreeCategory.getDefinition(__DEGREES_ATTRIBUTE_NAME), Optional.of(__DEGREES_ATTRIBUTE_NAME), SynchronizationContext.newInstance()))
                .map(ContentValue[].class::cast)
                .map(Arrays::stream)
                .orElseGet(Stream::empty)
                .map(degree -> degree.getContentIfExists())
                .flatMap(Optional::stream)
                .filter(degree -> _isInAnotherCategory(degree, degreeCategory))
                .map(degree -> _getErrorLabel(degree, degreeCategory))
                .forEach(result::addError);
        
        return result;
    }
    
    /**
     * Check if the current degree is in another category than the current category.
     * @param degree The current degree
     * @param degreeCategory The current category
     * @return <code>true</code> if it's in another degree category
     */
    protected boolean _isInAnotherCategory(Content degree, Content degreeCategory)
    {
        return _resolver.query(_getXPathQuery(degree, degreeCategory)).iterator().hasNext();
    }
    
    /**
     * Returns the XPath query to execute to obtain the list of categories except the current category in which the degree appears.
     * @param degree The degree
     * @param currentDegreeCategory The current degree category
     * @return An XPath query
     */
    protected String _getXPathQuery(Content degree, Content currentDegreeCategory)
    {
        Expression contentTypeExpr = new ContentTypeExpression(Operator.EQ, __DEGREES_CATEGORY_CONTENT_TYPE);
        Expression categoryExpr = new StringExpression(__DEGREES_ATTRIBUTE_NAME, Operator.EQ, degree.getId());
        String categoryUUID = StringUtils.substringAfter(currentDegreeCategory.getId(), "://");
        Expression notCurrentExpr = () -> JcrConstants.JCR_UUID + " != '" + categoryUUID + "'";
        Expression finalExpression = new AndExpression(contentTypeExpr, categoryExpr, notCurrentExpr);
        return ContentQueryHelper.getContentXPathQuery(finalExpression);
    }
    
    /**
     * Get the error label corresponding to the given degree and category.
     * @param degree The degree
     * @param degreeCategory The degree category
     * @return The error label as i18n
     */
    protected I18nizableText _getErrorLabel(Content degree, Content degreeCategory)
    {
        Map<String, I18nizableTextParameter> params = Map.of(
                "degree.title", new I18nizableText(StringUtils.defaultString(degree.getTitle())),
                "degree.code", new I18nizableText(StringUtils.defaultString(degree.getDataHolder().getValue("code"))),
                "degreeCategory.title", new I18nizableText(StringUtils.defaultString(degreeCategory.getTitle())),
                "degreeCategory.code", new I18nizableText(StringUtils.defaultString(degreeCategory.getDataHolder().getValue("code")))
        );
        return new I18nizableText("plugin.odf-ose", "PLUGINS_ODF_OSE_TABLE_REF_DEGREE_CATEGORY_VALIDATOR_UNCONSISTENCY_ERROR", params);
    }
}

