/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.export.AbstractOSEExport;
import org.ametys.odf.ose.export.OSEConstants;
import org.ametys.odf.ose.export.impl.odf.CourseExporter;
import org.ametys.odf.ose.export.impl.odf.HierarchyExporter;
import org.ametys.odf.ose.export.impl.odf.ProgramPartExporter;
import org.ametys.odf.ose.export.impl.odf.StepExporter;
import org.ametys.odf.ose.export.impl.odf.db.CheminPedagogiqueHelper;
import org.ametys.odf.ose.export.impl.odf.db.EffectifsHelper;
import org.ametys.odf.ose.export.impl.odf.db.ElementPedagogiqueHelper;
import org.ametys.odf.ose.export.impl.odf.db.EtapeHelper;
import org.ametys.odf.ose.export.impl.odf.db.LienHelper;
import org.ametys.odf.ose.export.impl.odf.db.NoeudHelper;
import org.ametys.odf.ose.export.impl.odf.db.ScenarioHelper;
import org.ametys.odf.ose.export.impl.odf.db.ScenarioLienHelper;
import org.ametys.odf.ose.export.impl.odf.db.ScenarioNoeudHelper;
import org.ametys.odf.ose.export.impl.odf.db.TypeInterventionEPHelper;
import org.ametys.odf.ose.export.impl.odf.db.VolumeHoraireEnsHelper;
import org.ametys.odf.program.AbstractProgramPart;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.odfpilotage.cost.CostComputationComponent;
import org.ametys.plugins.odfpilotage.cost.entity.CostComputationData;
import org.ametys.runtime.config.Config;

/**
 * This exports ODF values to the OSE database by SRC_* views.
 */
public class ODFExport extends AbstractOSEExport implements Serviceable
{
    /** The exporter of steps */
    protected StepExporter _stepExporter;
    /** The exporter of courses */
    protected CourseExporter _courseExporter;
    /** The exporter of program elements */
    protected ProgramPartExporter _programElementExporter;
    /** The exporter of hierarchy */
    protected HierarchyExporter _hierarchyExporter;
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    /** The cost computation component */
    protected CostComputationComponent _costComputationComponent;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _stepExporter = (StepExporter) manager.lookup(StepExporter.ROLE);
        _courseExporter = (CourseExporter) manager.lookup(CourseExporter.ROLE);
        _programElementExporter = (ProgramPartExporter) manager.lookup(ProgramPartExporter.ROLE);
        _hierarchyExporter = (HierarchyExporter) manager.lookup(HierarchyExporter.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _costComputationComponent = (CostComputationComponent) manager.lookup(CostComputationComponent.ROLE);
    }
    
    @Override
    public List<ParameterizableQuery> initializeDb()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.addAll(EtapeHelper.initialize());
        queries.addAll(ElementPedagogiqueHelper.initialize());
        queries.addAll(EffectifsHelper.initialize());
        queries.addAll(VolumeHoraireEnsHelper.initialize());
        queries.addAll(CheminPedagogiqueHelper.initialize());
        queries.addAll(TypeInterventionEPHelper.initialize());
        queries.addAll(NoeudHelper.initialize());
        queries.addAll(LienHelper.initialize());
        queries.addAll(ScenarioHelper.initialize());
        queries.addAll(ScenarioNoeudHelper.initialize());
        queries.addAll(ScenarioLienHelper.initialize());
        return queries;
    }
    
    @Override
    protected List<ParameterizableQuery> _populate(String ametysCatalog, Long oseCatalog)
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.addAll(_deleteOldData());
        queries.addAll(new Populator(oseCatalog).exportSteps(ametysCatalog));
        return queries;
    }
    
    private List<ParameterizableQuery> _deleteOldData()
    {
        List<ParameterizableQuery> queries = new ArrayList<>();
        queries.add(ScenarioLienHelper.deleteFrom());
        queries.add(ScenarioNoeudHelper.deleteFrom());
        queries.add(LienHelper.deleteFrom());
        queries.add(NoeudHelper.deleteFrom());
        queries.add(TypeInterventionEPHelper.deleteFrom());
        queries.add(CheminPedagogiqueHelper.deleteFrom());
        queries.add(VolumeHoraireEnsHelper.deleteFrom());
        queries.add(EffectifsHelper.deleteFrom());
        queries.add(ElementPedagogiqueHelper.deleteFrom());
        queries.add(EtapeHelper.deleteFrom());
        return queries;
    }
    
    private class Populator
    {
        private Set<String> _exportedObjectIds = new HashSet<>();
        private Long _oseCatalog;
        
        Populator(Long oseCatalog)
        {
            this._oseCatalog = oseCatalog;
        }
        
        List<ParameterizableQuery> exportSteps(String ametysCatalog)
        {
            CostComputationData costData = _getCostComputationData(ametysCatalog);
            
            List<ParameterizableQuery> queries = new ArrayList<>();
            for (Container step : _stepExporter.getStepsToExport(ametysCatalog))
            {
                if (_exportedObjectIds.add(step.getId()))
                {
                    queries.addAll(_stepExporter.getQueries(step, _oseCatalog));
                    queries.addAll(_exportChildren(step, costData));
                }
            }
            return queries;
        }
        
        private CostComputationData _getCostComputationData(String ametysCatalog)
        {
            String lang = Config.getInstance().getValue("odf.programs.lang");
            List<Program> programs = _odfHelper.getProgramItems(ProgramFactory.PROGRAM_CONTENT_TYPE, null, ametysCatalog, lang)
                    .stream()
                    .filter(Program.class::isInstance)
                    .map(Program.class::cast)
                    .collect(Collectors.toList());
            return _costComputationComponent.computeCostsOnPrograms(programs, false);
        }
        
        private List<ParameterizableQuery> _exportChildren(ProgramItem programItem, CostComputationData costData)
        {
            List<ParameterizableQuery> queries = new ArrayList<>();
            for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
            {
                if (_exportedObjectIds.add(child.getId()))
                {
                    if (child instanceof Course)
                    {
                        queries.addAll(_exportCourse((Course) child, costData));
                    }
                    else
                    {
                        // SubProgram, Container or CourseList
                        queries.addAll(_exportProgramPart((AbstractProgramPart) child, costData));
                    }
                    queries.addAll(_hierarchyExporter.getQueries(programItem, child, _oseCatalog));
                }
            }
            return queries;
        }
    
        private List<ParameterizableQuery> _exportProgramPart(AbstractProgramPart programPart, CostComputationData costData)
        {
            List<ParameterizableQuery> queries = new ArrayList<>();
            if (!programPart.hasDefinition(OSEConstants.NO_OSE_EXPORT_ATTRIBUTE_NAME) || !programPart.getValue(OSEConstants.NO_OSE_EXPORT_ATTRIBUTE_NAME, true, false))
            {
                queries.addAll(_programElementExporter.getQueries(programPart, _oseCatalog, costData));
                queries.addAll(_exportChildren(programPart, costData));
            }
            return queries;
        }
        
        private List<ParameterizableQuery> _exportCourse(Course course, CostComputationData costData)
        {
            List<ParameterizableQuery> queries = new ArrayList<>();
            if (!course.getValue(OSEConstants.NO_OSE_EXPORT_ATTRIBUTE_NAME, true, false))
            {
                queries.addAll(_courseExporter.getQueries(course, _oseCatalog, costData));
                queries.addAll(_exportChildren(course, costData));
            }
            return queries;
        }
    }
}

