/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.export.impl.odf.db;

import java.sql.Types;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang3.tuple.Pair;

import org.ametys.odf.ose.db.ParameterizableQuery;
import org.ametys.odf.ose.db.column.Column;
import org.ametys.odf.ose.db.column.DefaultColumn;
import org.ametys.odf.ose.db.column.ForeignKeyColumn;
import org.ametys.odf.ose.db.column.ScenarioIdColumn;
import org.ametys.odf.ose.db.column.SourceIdColumn;
import org.ametys.odf.ose.db.parameter.QueryParameter;
import org.ametys.odf.ose.db.parameter.StaticQueryParameter;
import org.ametys.odf.ose.db.parameter.ValuedQueryParameter;
import org.ametys.odf.ose.export.ExportUtils;

/**
 * Helper to build queries to manage the SCENARIO_LIEN table.
 */
public final class ScenarioLienHelper
{
    private static final String __TABLE_NAME = "SCENARIO_LIEN";
    
    private ScenarioLienHelper()
    {
        // Helper
    }

    /**
     * Queries to create the table and associated view.
     * @return List of queries for table and view creation
     */
    public static List<ParameterizableQuery> initialize()
    {
        Set<Column> columns = new HashSet<>();
        columns.add(new DefaultColumn("ID", "VARCHAR2(100 CHAR)", false, "SOURCE_CODE"));
        columns.add(new ForeignKeyColumn("LIEN_ID", "VARCHAR2(100 CHAR)", false, Pair.of("LIEN", "SOURCE_CODE"), false));
        columns.add(new DefaultColumn("ACTIF", "NUMBER(1)", false));
        columns.add(new DefaultColumn("POIDS", "FLOAT", false));
        columns.add(new DefaultColumn("CHOIX_MINIMUM", "NUMBER", true));
        columns.add(new DefaultColumn("CHOIX_MAXIMUM", "NUMBER", true));
        columns.add(new ScenarioIdColumn());
        columns.add(new SourceIdColumn());
        return ExportUtils.initializeTableAndView(__TABLE_NAME, columns);
    }

    /**
     * Query to delete all lines from the table.
     * @return The query for deletion
     */
    public static ParameterizableQuery deleteFrom()
    {
        return ExportUtils.deleteFromAmetys(__TABLE_NAME);
    }

    /**
     * Query to insert a line in the table.
     * @param code The code
     * @param weight The weight (1 in most case, can be 0 for optional lists and min/nbChildren for choice lists)
     * @param minChoice The min choice (only for choice lists), otherwise should be null
     * @param maxChoice The max choice (only for choice lists), otherwise should be null
     * @return The query to insert values in the table
     */
    public static ParameterizableQuery insertInto(String code, Double weight, Long minChoice, Long maxChoice)
    {
        List<QueryParameter> parameters = new ArrayList<>();
        parameters.add(new ValuedQueryParameter("ID", code, Types.VARCHAR));
        parameters.add(new ValuedQueryParameter("LIEN_ID", code, Types.VARCHAR));
        parameters.add(new StaticQueryParameter("ACTIF", String.valueOf(1)));
        parameters.add(new ValuedQueryParameter("POIDS", weight, Types.FLOAT));
        parameters.add(new ValuedQueryParameter("CHOIX_MINIMUM", minChoice, Types.NUMERIC));
        parameters.add(new ValuedQueryParameter("CHOIX_MAXIMUM", maxChoice, Types.NUMERIC));
        return ExportUtils.insertIntoAmetys(__TABLE_NAME, parameters);
    }
}
