/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.odf.ose.db.column;

import org.apache.commons.lang3.tuple.Pair;

/**
 * The definition of a column with a foreign key.
 * It extends {@link DefaultColumn} and add the notion of foreign key.
 * For integrity reasons, the AMETYS_* tables haven't the foreign, it's only used to retrieve values from the SRC_* views.
 */
public class ForeignKeyColumn extends DefaultColumn
{
    /** The foreign key: left is table name and right is column name of the referenced column */
    protected Pair<String, String> _foreignKey;
    /** If the foreign key needs to match the year */
    protected boolean _needYear;
    
    /**
     * Constructor.
     * @param name The name of the column
     * @param type The SQL type of the column
     * @param isNullable <code>true</code> if it's nullable
     * @param foreignKey The foreign key
     * @param needYear If the foreign key needs to match the year
     * @param viewName The name of the column in the associated SRC_* view
     */
    public ForeignKeyColumn(String name, String type, boolean isNullable, Pair<String, String> foreignKey, boolean needYear, String viewName)
    {
        super(name, type, isNullable, viewName);
        this._foreignKey = foreignKey;
        this._needYear = needYear;
    }
    
    /**
     * Constructor
     * @param name The name of the column (it will also be used for the associated SRC_* view)
     * @param type The SQL type of the column
     * @param isNullable <code>true</code> if it's nullable
     * @param foreignKey The foreign key
     * @param needYear If the foreign key needs to match the year
     */
    public ForeignKeyColumn(String name, String type, boolean isNullable, Pair<String, String> foreignKey, boolean needYear)
    {
        this(name, type, isNullable, foreignKey, needYear, name);
    }
    
    @Override
    public String getViewDescription()
    {
        StringBuilder sb = new StringBuilder();
        
        sb.append("(SELECT FKT.ID FROM ");
        sb.append(_foreignKey.getLeft());
        sb.append(" FKT WHERE FKT.");
        sb.append(_foreignKey.getRight());
        sb.append(" = TBL.");
        sb.append(_name);
        if (_needYear)
        {
            sb.append(" AND FKT.ANNEE_ID = TBL.ANNEE_ID");
        }
        sb.append(") AS ");
        sb.append(_viewName);
        
        return sb.toString();
    }
}
