
/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Actions for MCC workflow
 * @private
 */
Ext.define('Ametys.plugins.odf.pilotage.actions.MCCWorkflowActions', {
	singleton: true,
	
	/**
	 * Validates or invalidate rules
	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
	 * @param {Boolean} state the state of the button
	 */
	validateRules: function (controller, state)
	{
		var contentIds = controller.getContentIds();
		var revert = controller.isRevertAction();
        if (revert)
        {
            Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_REVERT_CONFIRM_TITLE}}", 
                        "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_REVERT_CONFIRM}}",
                        function(answer) {
                            if (answer == 'yes')
                            {
                                controller.serverCall (
						                'invalidateRules', 
						                [contentIds], 
						                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_REVERT_CONFIRM_TITLE}}"], 1), 
						                { refreshing: true, errorMessage: true }
						        );
                            }
                        },
                        this
                );
                
        }
        else
        {
            Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                dialogTitle: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_LABEL}}",
                dialogIconCls: "ametysicon-desktop-clipboard-list decorator-ametysicon-sign-raw-check",
                dateValue: new Date(),
                maxDate: new Date(),
                validateCallback: Ext.bind(this._validateRules, this, [controller, contentIds], 2)
            })
        }
	},
    
    _validateRules: function (date, comment, controller, contentIds)
    {
        controller.serverCall (
                'validateRules', 
                [contentIds, date, comment], 
                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_RULES_VALIDATED_LABEL}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    _updateMCCStatusCb: function (result, controller, contentIds, actionTitle)
    {
        if (result['error'] == "invalid-schedulable")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_SCHEDULABLE_ACTION_ERROR}}",
                msg: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_SCHEDULABLE_ACTION_ERROR_MSG}}",
                buttons: Ext.Msg.OK,
                icon: Ext.Msg.WARNING
            });
        }
        else
        {
            var successContentIds = result['success-contents'] || [];
            
            if (successContentIds.length > 0)
            {
                Ext.create("Ametys.message.Message", {
                    type: Ametys.message.Message.MODIFIED,
                    parameters: {major: true},
                    targets: {
                        id: Ametys.message.MessageTarget.CONTENT,
                        parameters: {
                            ids: successContentIds
                        }
                    }
                });
            }
            
            var noRightContents = result['noright-contents'] || [];
            var invalidContents = result['invalidmccstatus-contents'] || [];
            var lockedContents = result['locked-contents'] || [];
            
            var warnMsg = "";
            if (noRightContents.length > 0)
            {
                warnMsg += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_NO_RIGHT_ERROR}}";
                warnMsg += "<ul>";
                for(var i=0; i < noRightContents.length; i++)
                {
                    warnMsg += "<li>" + noRightContents[i].title + "</li>";
                }
                warnMsg += "</ul>";
            }
            
            if (invalidContents.length > 0)
            {
                warnMsg += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_INVALIDMCCSTATUS_ERROR}}";
                warnMsg += "<ul>";
                for(var i=0; i < invalidContents.length; i++)
                {
                    warnMsg += "<li>" + invalidContents[i].title + "</li>";
                }
                warnMsg += "</ul>";
            }
            
            if (lockedContents.length > 0)
            {
                warnMsg += "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_LOCKED_ERROR}}";
                warnMsg += "<ul>";
                for(var i=0; i < lockedContents.length; i++)
                {
                    warnMsg += "<li>" + lockedContents[i].title + "</li>";
                }
                warnMsg += "</ul>";
            }
            
            var schedulable = result['schedulable-id'];
            if (warnMsg)
            {
                if (successContentIds.length > 0 && schedulable)
                {
                    warnMsg += "<br/>" + Ext.String.format("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_ERROR_MSG}}", successContentIds.length)
                }
                
                Ametys.Msg.show({
                    title: actionTitle,
                    msg: warnMsg,
                    buttons: Ext.Msg.OK,
                    icon: Ext.Msg.WARNING
                });
            }
            
            if (schedulable)
            {
                this._checkTaskState(schedulable, {
                    taskLabel: result['schedulable-label'],
                });
            }
        }
    },
    
    /**
     * Validates or invalidate MCC for mention level
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     * @param {Boolean} state the state of the button
     */
    validateMCC: function (controller, state)
    {
        var contentIds = controller.getContentIds();
        var revert = controller.isRevertAction();
        if (revert)
        {
            Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_REVERT_CONFIRM_TITLE}}", 
                        "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_REVERT_CONFIRM}}",
                        function(answer) {
                            if (answer == 'yes')
                            {
                                controller.serverCall (
                                        'invalidateMCC', 
                                        [contentIds], 
                                        Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds,"{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_REVERT_CONFIRM_TITLE}}"], 1), 
                                        { refreshing: true, errorMessage: true }
                                );
                            }
                        },
                        this
                );
                
        }
        else
        {
            Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                dialogTitle: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_LABEL}}",
                dialogIconCls: "ametysicon-desktop-clipboard-list decorator-ametysicon-sign-raw-check",
                dateValue: new Date(),
                maxDate: new Date(),
                validateCallback: Ext.bind(this._validateMCC, this, [controller, contentIds], 2)
            })
        }
    },
    
    _validateMCC: function (date, comment, controller, contentIds)
    {
        controller.serverCall (
                'validateMCC', 
                [contentIds, date, comment], 
                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_VALIDATED_LABEL}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    /**
     * Validates or invalidate the MCC for orgunit level
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     * @param {Boolean} state the state of the button
     */
    validateOrgunitMCC: function (controller, state)
    {
        var contentIds = controller.getContentIds();
        
        var revert = controller.isRevertAction();
        if (revert)
        {
            Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_REVERT_CONFIRM_TITLE}}", 
                        "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_REVERT_CONFIRM}}",
                        function(answer) {
                            if (answer == 'yes')
                            {
                                controller.serverCall (
						                'invalidateOrgunitMCC', 
						                [contentIds], 
						                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_REVERT_CONFIRM_TITLE}}"], 1), 
						                { refreshing: true, errorMessage: true }
						        );
                            }
                        },
                        this
                );
                
        }
        else
        {
            var me = this;
            controller.serverCall (
                    'getMinDateForMCCOrgUnitValidation', 
                    [contentIds], 
                    function (date) {
                        Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                            dialogTitle: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_LABEL}}",
                            dialogIconCls: "ametysicon-desktop-clipboard-list decorator-ametysicon-sign-raw-check",
                            dateValue: new Date(),
                            minDate: date ? new Date(date) : null,
                            maxDate: new Date(),
                            validateCallback: Ext.bind(me._validateOrgunitMCC, me, [controller, contentIds], 2)
                        })
                    }
            );
        }
    },
    
    _validateOrgunitMCC: function (date, comment, controller, contentIds)
    {
        controller.serverCall (
                'validateOrgunitMCC', 
                [contentIds, date, comment], 
                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_MCC_ORGUNIT_VALIDATED_LABEL}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    /**
     * Validates MCC
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     * @param {Boolean} state the state of the button
     */
    validateMCCForCVFU: function (controller, state)
    {
        var contentIds = controller.getContentIds();

        var revert = controller.isRevertAction();
        if (revert)
        {
            Ametys.Msg.confirm("{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_REVERT_CONFIRM_TITLE}}", 
                    "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_REVERT_CONFIRM}}",
                    function(answer) {
                       if (answer == 'yes') 
                       {
                            controller.serverCall (
			                    'invalidateMCCForCVFU', 
			                    [contentIds], 
			                    Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_REVERT_CONFIRM_TITLE}}"], 1), 
			                    { refreshing: true, errorMessage: true }
				            );
                       }
                    },
                    this
            );
        }
        else
        {
            var me = this;
            controller.serverCall (
                    'getMinDateForMCCCFVUValidation', 
                    [contentIds], 
                    function (date) {
                        Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                            dialogTitle: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_LABEL}}",
                            dialogIconCls: "odficon-control",
                            hintMsg: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_HINT}}",
                            dateValue: new Date(),
                            minDate: date ? new Date(date) : null,
                            maxDate: new Date(),
                            validateCallback: Ext.bind(me._validateMCCForCVFU, me, [controller, contentIds], 2)
                        })
                    }
            );
        }
    },
    
    _validateMCCForCVFU: function (date, comment, controller, contentIds)
    {
        controller.serverCall (
                'validateMCCForCVFU', 
                [contentIds, date, comment, Ametys.getAppParameters()], 
                Ext.bind(this._updateMCCStatusCb, this, [controller, contentIds, "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_CFVU_VALIDATED_LABEL}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    /**
     * Validates the MCC for orgunit level for all years of the selected program
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     * @param {Boolean} state the state of the button
     */
    validateOrgunitMCCForChildYears: function (controller, state)
    {
        var programIds = controller.getContentIds();
        
        if (programIds.length > 0)
        {
            var me = this;
            controller.serverCall (
                    'getMinDateForMCCOrgUnitValidation', 
                    [programIds[0]], 
                    function (date) {
                        Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                            dialogTitle: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_ORGUNIT_VALIDATION_LABEL}}",
                            dialogIconCls: "ametysicon-desktop-clipboard-list decorator-ametysicon-sign-raw-check",
                            dateValue: new Date(),
                            minDate: date ? new Date(date) : null,
                            maxDate: new Date(),
                            validateCallback: Ext.bind(me._validateOrgunitMCCForChildYears, me, [controller, programIds[0]], 2)
                        })
                    }
            );
        }
    },
    
    
    _validateOrgunitMCCForChildYears: function (date, comment, controller, programId)
    {
        controller.serverCall (
                'validateOrgunitMCC', 
                [programId, date, comment], 
                Ext.bind(this._updateProgramMCCStatusCb, this, [controller, [programId], "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_ORGUNIT_VALIDATION_LABEL}}", "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_ORGUNIT_VALIDATION_SUCCESS}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    /**
     * Validates the MCC for CFVU level for all years of the selected program
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     * @param {Boolean} state the state of the button
     */
    validateCFVUMCCForChildYears: function (controller, state)
    {
        var programIds = controller.getContentIds();
        
        if (programIds.length > 0)
        {
            var me = this;
            controller.serverCall (
                    'getMinDateForMCCCFVUValidation', 
                    [programIds[0]], 
                    function (date) {
                        Ametys.plugins.odf.pilotage.helper.ValidateDialog.open({
                            dialogTitle: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_CFVU_VALIDATION_LABEL}}",
                            dialogIconCls: "ametysicon-desktop-clipboard-list decorator-ametysicon-sign-raw-check",
                            dateValue: new Date(),
                            hintMsg: "{{i18n plugin.odf-pilotage:PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_CFVU_HINT}}",
                            minDate: date ? new Date(date) : null,
                            maxDate: new Date(),
                            validateCallback: Ext.bind(me._validateCFVUMCCForChildYears, me, [controller, programIds[0]], 2)
                        })
                    }
            );
        }
    },
    
    
    _validateCFVUMCCForChildYears: function (date, comment, controller, programId)
    {
        controller.serverCall (
                'validateMCCForCVFU', 
                [programId, date, comment, Ametys.getAppParameters()], 
                Ext.bind(this._updateProgramMCCStatusCb, this, [controller, [programId], "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_CFVU_VALIDATION_LABEL}}", "{{i18n PLUGINS_ODF_PILOTAGE_MCC_WORKFLOW_PROGRAM_MCC_CFVU_VALIDATION_SUCCESS}}"], 1), 
                { refreshing: true, errorMessage: true }
        );
    },
    
    _updateProgramMCCStatusCb: function (result, controller, programIds, actionTitle, successMsg)
    {
        var successContentIds = result['success-contents'] || [];
        
        if (successContentIds.length > 0)
        {
            Ametys.cms.content.ContentDAO.getContents(programIds, function(contents) {
                Ext.Array.forEach(contents, function(content) {
                    var infoMsg = Ext.String.format(successMsg, successContentIds.length, content.getTitle());
                     Ametys.notify({
		                type: 'info',
		                iconGlyph: 'ametysicon-sign-raw-check',
		                title: actionTitle,
		                description: infoMsg
		            });
                })
            })
        }
        
        this._updateMCCStatusCb(result, controller, programIds, actionTitle);
    },
    
    /**
     * Check the task state
     * @param {String} taskId the task id
     * @param {Object} args the given arguments
     * @private
     */
    _checkTaskState: function(taskId, args)
    {
        Ametys.plugins.core.schedule.Scheduler.getTasksInformation(
            [[taskId]], 
            this._checkTaskStateCb, 
            {
                errorMessage: Ext.String.format("{{i18n plugin.core-ui:PLUGINS_CORE_UI_TASKS_ADD_TASK_BUTTON_CONTROLLER_CHECK_STATE_ERROR}}", args.taskLabel),
                scope: this, 
                arguments: args
            }
        );
    },
    
    /**
     * Callback after checking the task state
     * @param {Object} response the response
     * @param {Object} args the given arguments
     * @private
     */
    _checkTaskStateCb: function(response, args)
    {
        if (response.length > 0)
        {
            var task = response[0];
            if (task.getState() == "running")
            {
                Ext.defer(this._checkTaskState, 1000, this, [task.getId(), args]);
            }
            else if (task.getState() == "success")
            {
               Ametys.notify({
                    type: 'info',
                    title: args.taskLabel,
                    description: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_SCHEDULABLE_ENDED_SUCCESS}}"
                });
            }
            else if (task.getState() == "failure")
            {
               Ametys.notify({
                    type: 'error',
                    title: args.taskLabel,
                    description: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_CONTAINER_SCHEDULABLE_ENDED_FAILURE}}"
                });
            }
        }
    },
    
    /**
     * Action function to be called by the controller.
     * Will open the dialog box to edit validated MCC, only permitted action is deleting MCC's pdf because addition is blocked by unavailable mandatory fields
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
     */
    seeValidatedMCC: function(controller)
    {
        var contentIds = controller.getContentIds()
        if (contentIds.length > 0)
        {
            var targets = controller.getMatchingTargets().filter(t => t.getParameters().id == contentIds[0])
            if (targets.length > 0)
            {
                var target = targets[0];
                var rights = target.getParameters().rights || [];
                var disallowEdition = !Ext.Array.contains(rights, "ODF_Pilotage_Delete_MCC_Validated_Rights");
                
                var openParams = {
                    content: target.getParameters().content,
                    editWorkflowActionId: 222227,
                    iconCls: 'ametysicon-interface decorator-ametysicon-body-part-eye',
                    viewName: "mcc-validated-pdf",
                    width: 800,
                    labelAlign: 'right',
                    title: "{{i18n PLUGINS_ODF_PILOTAGE_MCC_VALIDATED_PDF_SEE_DIALOG_TITLE}} "  + "\"" + target.getParameters().title + "\"",
                    disallowEdition: disallowEdition
                }
            
                Ametys.cms.uihelper.EditContent.open(openParams, null, this);
            }
        }
    }
    
});
