/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * Helper for step holder tree indicator
 */
Ext.define('Ametys.plugins.odf.pilotage.helper.CourseStepHolderTreeIndicator', {
    singleton: true,
   
    /**
     * Determine if the indicator is available for the record.
     * Returns true if the indicator data are not null and the step holder is not part of the course's path, or if there is an error on step holder
     * @param {String} id the id of indicator
     * @param {Ext.data.Record} record the record
     * @return {Boolean} true if the indicator data are not null and the step holder is not part of the course's path.
     */
    matchIndicator: function(id, record)
    {
        var indicatorData = record.get(id);
        if (indicatorData)
        {
            return indicatorData.error 
                    || !record.getPath("contentId", ';').includes(indicatorData.stepHolderId)
            
        }
    },
    
    /**
     * Add the possible steps holder if there are any.
     * @param {Ametys.plugins.odf.tree.ODFContentsTreePanel} tree the tree
     * @param {String} id the id of indicator
     * @param {Ext.data.Record} record the record
     * @return {String|HTML} The display values
     */
    applyIndicator: function (tree, id, record)
    {
        var indicatorData = record.get(id);
        
        var tooltip = indicatorData.tooltip || '';
        var availableStepHolders = indicatorData.availableStepHolders;
        if (availableStepHolders && availableStepHolders.length > 1)
        {
            tooltip += `<ul>`;
            for (let stepHolder of availableStepHolders)
            {
                tooltip += `<li>${stepHolder}</li>`;
            }
            tooltip += `</ul>`;
        }
        
        return `<span data-qtip="${tooltip}" class="${tree.indicatorCls} ${indicatorData.cssClass || ''}">${indicatorData.text || ''}</span>`;
    }
});