/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ametys.core.ui.Callable;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper.MCCWorkflowStep;

/**
 * Client side element for MCC workflow buttons on a {@link Program}
 */
public class ProgramMCCWorkflowClientSideElement extends MCCWorkflowClientSideElement
{
    /**
     * Get informations on program's state
     * @param programId the id of program
     * @param mccStatus the MCC status
     * @return informations on program's' state
     */
    @Callable
    public Map<String, Object> getStatus(String programId, String mccStatus)
    {
        Program program = _resolver.resolveById(programId);
        
        Set<Container> childYears = _pilotageHelper.getYears(program);
        
        if (childYears.isEmpty())
        {
            // Program has no years
            return Map.of("no-years", true);
        }
        
        switch (MCCWorkflowStep.valueOf(mccStatus))
        {
            case MCC_ORGUNIT_VALIDATED:
                if (childYears.stream().allMatch(_mccWorkflowHelper::isMCCOrgunitValidated))
                {
                    // All child years are orgunit validated
                    return Map.of("all-active", true);
                }
                else if (!childYears.stream().anyMatch(_mccWorkflowHelper::canValidateOrgUnitMCC))
                {
                    // None child years can be validated
                    return Map.of("no-available", true);
                }
                break;
            case CFVU_MCC_VALIDATED:
                if (childYears.stream().allMatch(_mccWorkflowHelper::isMCCCFVUValidated))
                {
                    // All child years are CFVU validated
                    return Map.of("all-active", true);
                }
                else if (!childYears.stream().anyMatch(_mccWorkflowHelper::canValidateCFVUMCC))
                {
                    // None child years can be validated
                    return Map.of("no-available", true);
                }
                break;
            default:
                throw new IllegalArgumentException("Unexpected value for MCC status : " + mccStatus);
        }
        
        Map<String, Object> results = new HashMap<>();
        results.put("active-contents", new ArrayList<>());
        results.put("allright-contents", new ArrayList<>());
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> activeContents = (List<Map<String, Object>>) results.get("active-contents");
        
        for (Container year : childYears)
        {
            _getContainerStatus(year, mccStatus, activeContents);
        }
        
        Map<String, Object> contentParams = getContentDefaultParameters (program);
        contentParams.put("description", _getAllRightDescription(program));
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        allrightContents.add(contentParams);
        
        return results;
    }
    
    private void _getContainerStatus(Container container, String mccStatus, List<Map<String, Object>> activeContents)
    {
        Map<String, Object> defaultContentParams = getContentDefaultParameters(container);
        
        switch (MCCWorkflowStep.valueOf(mccStatus))
        {
            case MCC_ORGUNIT_VALIDATED:
                if (_mccWorkflowHelper.isMCCOrgunitValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                break;
            case CFVU_MCC_VALIDATED:
                if (_mccWorkflowHelper.isMCCCFVUValidated(container))
                {
                    Map<String, Object> activeContentParams = new HashMap<>(defaultContentParams);
                    activeContentParams.put("description", _getActiveDescription(container));
                    activeContents.add(activeContentParams);
                }
                break;
            default:
                throw new IllegalArgumentException("Unexpected value for MCC status : " + mccStatus);
        }
    }
    
    /**
     * Get minimun date for MCC orgunit validation
     * @param programItemId the id of root program
     * @return the min date
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public LocalDate getMinDateForMCCOrgUnitValidation(String programItemId)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _pilotageHelper.getYears(programItem)
                .stream()
                .filter(_mccWorkflowHelper::canValidateOrgUnitMCC)
                .map(Container::getId)
                .toList();
        
        return super.getMinDateForMCCOrgUnitValidation(childYearIds);
    }
    
    /**
     * Get minimun date for MCC CFVU validation
     * @param programItemId the id of root program
     * @return the min date
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public LocalDate getMinDateForMCCCFVUValidation(String programItemId)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _pilotageHelper.getYears(programItem)
                .stream()
                .filter(_mccWorkflowHelper::canValidateCFVUMCC)
                .map(Container::getId)
                .toList();
        
        return super.getMinDateForMCCCFVUValidation(childYearIds);
    }
    
    /**
     * Validate the MCC to orgunit level for all child years of a program
     * @param programItemId the id of root program
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateOrgunitMCC(String programItemId, String validationDate, String comment)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _pilotageHelper.getYears(programItem)
                .stream()
                .filter(c -> !_mccWorkflowHelper.isMCCOrgunitValidated(c))
                .map(Container::getId)
                .toList();
        
        return validateOrgunitMCC(childYearIds, validationDate, comment);
    }
    
    /**
     * Validate the MCC to orgunit level for all child years of a program
     * @param programItemId the id of root program
     * @param validationDate the date of validation
     * @param comment the optional comment
     * @param contextualParameters the contextual parameters
     * @return the result
     */
    @Callable (rights = Callable.SKIP_BUILTIN_CHECK)
    public Map<String, Object> validateMCCForCVFU(String programItemId, String validationDate, String comment, Map<String, Object> contextualParameters)
    {
        TraversableProgramPart programItem = _resolver.resolveById(programItemId);
        
        List<String> childYearIds = _pilotageHelper.getYears(programItem)
                .stream()
                .filter(c -> !_mccWorkflowHelper.isMCCCFVUValidated(c))
                .map(Container::getId)
                .toList();
        
        return validateMCCForCVFU(childYearIds, validationDate, comment, contextualParameters);
        
    }
}
