/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.consistency;

import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * Enumerator for {@link ConsistencyAnalysis}
 */
public class AnalysisEnumerator implements Enumerator<String>, Serviceable, Configurable
{
    /** The analysis extension point */
    protected AnalysisExtensionPoint _analysisEP;
    
    /**
     * <code>true</code> to have the all option.
     * Default value: <code>true</code>
     **/
    protected boolean _allOption;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _analysisEP = (AnalysisExtensionPoint) smanager.lookup(AnalysisExtensionPoint.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _allOption = configuration.getChild("enumeration").getChild("custom-enumerator").getChild("allOption").getValueAsBoolean(true);
    }
    
    public Map<String, I18nizableText> getTypedEntries() throws Exception
    {
        Map<String, I18nizableText> entries = _analysisEP.getExtensionsIds()
            .stream()
            .map(_analysisEP::getExtension)
            .collect(Collectors.toMap(ConsistencyAnalysis::getId, ConsistencyAnalysis::getLabel));
        
        // All analyses
        _handleAllOptionEntry(entries);
        
        return entries;
    }
    
    /**
     * Add the all option entry to the entry list if necessary
     * @param entries The enumerator entries
     */
    protected void _handleAllOptionEntry(Map<String, I18nizableText> entries)
    {
        if (_allOption)
        {
            entries.put("", new I18nizableText("plugin.cms", "WIDGET_COMBOBOX_ALL_OPTIONS"));
        }
    }

    @Override
    public I18nizableText getEntry(String value) throws Exception
    {
        ConsistencyAnalysis analysis = _analysisEP.getExtension(value);
        return analysis.getLabel();
    }
}
