/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.content.compare.ContentComparator;
import org.ametys.cms.content.compare.ContentComparatorChange;
import org.ametys.cms.content.compare.ContentComparatorResult;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.plugins.odfpilotage.report.impl.mcc.MCCProgramItemTree;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitMCCDiffReportSchedulable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.runtime.model.ModelItem;

/**
 * The compare MCC catalog report (based on MCC report).
 */
public class MCCDiffReport extends AbstractMCCReport
{
    /** The key for the old catalog */
    public static final String PARAMETER_CATALOG_OLD = "catalogOld";
    
    private ContentComparator _contentComparator;

    /**
     * Change type
     */
    protected enum ChangeType
    {
        /**
         * The metadata is added in the new content
         */
        ADDED,
        /**
         * The metadata is removed in the new content
         */
        REMOVED,
        /**
         * The metadata is modified in the old content
         */
        MODIFIED_OLD,
        /**
         * The metadata is modified in the new content
         */
        MODIFIED_NEW
    }
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        return "mccdiff";
    }

    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable instanceof OrgUnitMCCDiffReportSchedulable;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _contentComparator = (ContentComparator) manager.lookup(ContentComparator.ROLE);
    }
    
    @Override
    protected MCCProgramItemTree _createMCCTreeFromProgramItem(ProgramItem programItem, Map<String, String> reportParameters)
    {
        ProgramItem oldProgramItem = _findOldProgramItem(programItem, reportParameters.get(PARAMETER_CATALOG_OLD));
        ContentComparatorResult changes = _getObjectChanges(oldProgramItem, programItem);
        MCCProgramItemTree programItemTree = new MCCProgramItemTree(programItem, changes);
        populateMCCTree(programItemTree);
        return programItemTree;
    }
    
    @Override
    protected void populateMCCTree(MCCProgramItemTree tree)
    {
        ProgramItem oldProgramItem = (ProgramItem) tree.getChange().getContent1();
        ProgramItem newProgramItem = (ProgramItem) tree.getChange().getContent2();
        
        if (oldProgramItem == null)
        {
            // Only new
            List<ProgramItem> children = _odfHelper.getChildProgramItems(newProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(null, child);
                MCCProgramItemTree childTree = tree.addChild(child, changes);
                populateMCCTree(childTree);
            }
        }
        else if (newProgramItem == null)
        {
            // Only old
            List<ProgramItem> children = _odfHelper.getChildProgramItems(oldProgramItem);
            for (ProgramItem child : children)
            {
                ContentComparatorResult changes = _getObjectChanges(child, null);
                MCCProgramItemTree childTree = tree.addChild(child, changes);
                populateMCCTree(childTree);
            }
        }
        else
        {
            List<ProgramItem> oldChildren = _odfHelper.getChildProgramItems(oldProgramItem);
            List<ProgramItem> newChildren = _odfHelper.getChildProgramItems(newProgramItem);

            // Add new program items
            for (ProgramItem newChild : newChildren)
            {
                ProgramItem currentOldChild = null;
                Class<? extends ProgramItem> newChildClass = newChild.getClass();
                String newChildCode = newChild.getCode();
                for (ProgramItem oldChild : oldChildren)
                {
                    if (oldChild.getCode().equals(newChildCode) && oldChild.getClass().equals(newChildClass))
                    {
                        currentOldChild = oldChild;
                        // Decrease the size of old children to do the next operations faster
                        oldChildren.remove(oldChild);
                        break;
                    }
                }

                ContentComparatorResult changes = _getObjectChanges(currentOldChild, newChild);
                MCCProgramItemTree childTree = tree.addChild(newChild, changes);
                populateMCCTree(childTree);
            }
            
            // Add old program items
            for (ProgramItem oldChild : oldChildren)
            {
                ContentComparatorResult changes = _getObjectChanges(oldChild, null);
                MCCProgramItemTree childTree = tree.addChild(oldChild, changes);
                populateMCCTree(childTree);
            }
        }
    }

    private ContentComparatorResult _getObjectChanges(ProgramItem oldProgramItem, ProgramItem newProgramItem)
    {
        return (oldProgramItem instanceof Course || newProgramItem instanceof Course)
                ? _getCourseChanges((Course) oldProgramItem, (Course) newProgramItem)
                : new ContentComparatorResult((Content) oldProgramItem, (Content) newProgramItem);
    }
    
    /**
     * Compare two courses.
     * @param oldCourse The first course to compare (the old one)
     * @param newCourse The second course to compare (the new one)
     * @return A {@link ContentComparatorResult}, or null if an exception occured
     */
    private ContentComparatorResult _getCourseChanges(Course oldCourse, Course newCourse)
    {
        if (oldCourse == null || newCourse == null)
        {
            return new ContentComparatorResult(oldCourse, newCourse);
        }

        ContentComparatorResult changes = null;
        try
        {
            changes = _contentComparator.compare(oldCourse, newCourse, "mcc");
            if (getLogger().isDebugEnabled() && !changes.areEquals())
            {
                getLogger().debug("Différence trouvée pour l'ELP {} - {}", oldCourse.getCode(), oldCourse.getTitle());
            }
        }
        catch (AmetysRepositoryException | IOException e)
        {
            getLogger().error("Une erreur est survenue pour l'ELP {} - {}", oldCourse.getCode(), oldCourse.getTitle(), e);
        }
        
        return changes;
    }
    
    /**
     * Find the equivalent content in the new catalog
     * @param <T> Type of content to find
     * @param content Content in the current catalog
     * @return New equivalent content or null
     */
    private <T extends ProgramItem> T _findOldProgramItem(T content, String oldCatalog)
    {
        return _odfHelper.getProgramItem(content, oldCatalog, content.getLanguage());
    }
    
    private Optional<ModelAwareRepeater> _getRepeater(Content content, String sessionName)
    {
        return Optional.ofNullable(content)
            .map(c -> c.getRepeater(sessionName));
    }
    
    private Set<Integer> _getEntryPositions(Optional<ModelAwareRepeater> repeater)
    {
        return repeater
            .map(ModelAwareRepeater::getEntries)
            .map(List::stream)
            .orElseGet(Stream::of)
            .map(ModelAwareRepeaterEntry::getPosition)
            .collect(Collectors.toSet());
    }
    
    private ModelAwareRepeaterEntry _getEntry(Optional<ModelAwareRepeater> repeater, Integer position)
    {
        return repeater.map(s -> s.getEntry(position)).orElse(null);
    }

    @Override
    protected void _saxMCCs(ContentHandler handler, Course course, MCCProgramItemTree tree) throws SAXException
    {
        XMLUtils.startElement(handler, "mcc");
        
        // Generate SAX events for MCC sessions
        _saxSession(handler, tree, FIRST_SESSION_NAME);
        _saxSession(handler, tree, SECOND_SESSION_NAME);
        
        XMLUtils.endElement(handler, "mcc");
    }
    
    private void _saxSession(ContentHandler handler, MCCProgramItemTree tree, String sessionName) throws SAXException
    {
        ContentComparatorResult changes = tree.getChange();
        Content oldContent = changes.getContent1();
        Content newContent = changes.getContent2();
        
        ModelItem sessionDefinition = _sessionsView.getModelViewItem(sessionName).getDefinition();
        
        boolean oldContentSessionEnabled = oldContent != null && !_disableConditionsEvaluator.evaluateDisableConditions(sessionDefinition, sessionName, oldContent);
        boolean newContentSessionEnabled = newContent != null && !_disableConditionsEvaluator.evaluateDisableConditions(sessionDefinition, sessionName, newContent);
        
        if (oldContentSessionEnabled || newContentSessionEnabled)
        {
            XMLUtils.startElement(handler, sessionName);
    
            Optional<ModelAwareRepeater> oldSession = oldContentSessionEnabled ? _getRepeater(oldContent, sessionName) : Optional.empty();
            Optional<ModelAwareRepeater> newSession = newContentSessionEnabled ? _getRepeater(newContent, sessionName) : Optional.empty();
            
            // Concaténation des positions de l'ancien et du nouveau repeater
            Set<Integer> allEntryPositions = new TreeSet<>();
            allEntryPositions.addAll(_getEntryPositions(oldSession));
            allEntryPositions.addAll(_getEntryPositions(newSession));
    
            if (!allEntryPositions.isEmpty())
            {
                for (Integer entryPosition : allEntryPositions)
                {
                    ModelAwareRepeaterEntry oldEntry = _getEntry(oldSession, entryPosition);
                    ModelAwareRepeaterEntry newEntry = _getEntry(newSession, entryPosition);
                    
                    if (oldEntry == null)
                    {
                        /* Entrée ajoutée */
                        _saxSessionEntry(handler, newEntry, newContent, ChangeType.ADDED);
                    }
                    else if (newEntry == null)
                    {
                        /* Entrée supprimée */
                        _saxSessionEntry(handler, oldEntry, oldContent, ChangeType.REMOVED);
                    }
                    else if (changes.areEquals() || !_hasChanges(changes, sessionName, entryPosition))
                    {
                        /* Pas de changement */
                        _saxSessionEntry(handler, newEntry, newContent, null);
                    }
                    else if (_hasSameNature(oldEntry, newEntry))
                    {
                        /* Entrée modifiée (même natures) */
                        _saxSessionEntry(handler, oldEntry, oldContent, ChangeType.MODIFIED_OLD);
                        _saxSessionEntry(handler, newEntry, newContent, ChangeType.MODIFIED_NEW);
                    }
                    else
                    {
                        /* Entrée modifiée (natures différentes) */
                        _saxSessionEntry(handler, oldEntry, oldContent, ChangeType.REMOVED);
                        _saxSessionEntry(handler, newEntry, newContent, ChangeType.ADDED);
                    }
                }
            }
            
            XMLUtils.endElement(handler, sessionName);
        }
    }
    
    private void _saxSessionEntry(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent, ChangeType changeType) throws SAXException
    {
        // start entry
        AttributesImpl entryAttrs = new AttributesImpl();
        entryAttrs.addCDATAAttribute("name", String.valueOf(sessionEntry.getPosition()));
        if (changeType != null)
        {
            entryAttrs.addCDATAAttribute("changeType", changeType.name().toLowerCase());
        }
        XMLUtils.startElement(handler, "entry", entryAttrs);
        
        _saxSessionEntryDetails(handler, sessionEntry, rootContent);
        
        // end entry
        XMLUtils.endElement(handler, "entry");
    }
    
    private boolean _hasSameNature(ModelAwareRepeaterEntry oldEntry, ModelAwareRepeaterEntry newEntry)
    {
        ContentValue nature1 = oldEntry.getValue("natureEnseignement");
        ContentValue nature2 = newEntry.getValue("natureEnseignement");
        return Objects.equals(nature1, nature2);
    }
    
    private boolean _hasChanges(ContentComparatorResult result, String sessionName, Integer entryPosition)
    {
        String beginMetadataPath = sessionName + "[" + entryPosition + "]" + ModelItem.ITEM_PATH_SEPARATOR;
        for (ContentComparatorChange change : result.getChanges())
        {
            if (change.getPath().startsWith(beginMetadataPath))
            {
                return true;
            }
        }
        return false;
    }
    
    @Override
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxGlobalInformations(handler, programItem, reportParameters);
        
        String oldCatalog = reportParameters.get(PARAMETER_CATALOG_OLD);
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", oldCatalog);
        XMLUtils.createElement(handler, "catalogOld", attrs, _catalogsManager.getCatalog(oldCatalog).getTitle());
    }
}
