/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.rule.workflow;

import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.holder.group.ModifiableIndexableRepeater;
import org.ametys.cms.data.holder.group.ModifiableIndexableRepeaterEntry;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.workflow.EnhancedFunction;

/**
 * Abstract Workflow function to create unique code for thematics and rules.
 */
public abstract class AbstractCodeEditionFunction extends AbstractContentWorkflowComponent implements EnhancedFunction
{
    /**
     * Set rules code
     * @param content the content holding the rule
     * @param thematicDataHolder the thematic data holder
     * @param ruleRepeaterName the rule repeater name
     */
    protected void _setRulesCode(ModifiableContent content, ModifiableModelAwareDataHolder thematicDataHolder, String ruleRepeaterName)
    {
        List<? extends ModifiableIndexableRepeaterEntry> filteredEntries = Optional.ofNullable(thematicDataHolder)
                .map(c -> c.<ModifiableIndexableRepeater>getValue(ruleRepeaterName))
                .map(ModifiableIndexableRepeater::getEntries)
                .map(List::stream)
                .orElseGet(Stream::of)
                .filter(e -> StringUtils.isEmpty(e.getValue(RulesManager.RULE_CODE)))
                .toList();
        
        if (!filteredEntries.isEmpty())
        {
            String thematicCode = thematicDataHolder.getValue(RulesManager.THEMATIC_CODE);
            String numberAttribute = thematicCode + RulesManager.RULES_NB_SUFFIX;
            Long ruleNumber = content.getInternalDataHolder().getValue(numberAttribute, 0L);
            
            for (ModifiableIndexableRepeaterEntry entry : filteredEntries)
            {
                ruleNumber++;
                entry.setValue(RulesManager.RULE_CODE, thematicCode + "-" + ruleNumber);
            }
            
            content.getInternalDataHolder().setValue(numberAttribute, ruleNumber);
        }
    }
}
