/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.schedulable;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.PilotageReport.PilotageReportTarget;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import jakarta.mail.MessagingException;

/**
 * {@link Schedulable} for pilotage report.
 */
public abstract class AbstractReportSchedulable extends AbstractStaticSchedulable implements Initializable
{
    /** The key for the extension ID */
    public static final String JOBDATAMAP_EXTENSION_ID_KEY = "extensionId";
    
    /** The key for the output format */
    public static final String JOBDATAMAP_OUTPUT_FORMAT_KEY = "outputFormat";
    
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    
    /** The report helper */
    protected ReportHelper _reportHelper;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The I18N utils */
    protected I18nUtils _i18nUtils;

    private String _mailFrom;
    
    public void initialize() throws Exception
    {
        _mailFrom = Config.getInstance().getValue("smtp.mail.from");
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        for (Configuration paramConf : configuration.getChild("parameters").getChildren("param"))
        {
            if (paramConf.getAttribute("id").equals(AbstractReportSchedulable.JOBDATAMAP_EXTENSION_ID_KEY))
            {
                Configuration customEnumConf = Optional.of(paramConf)
                    .map(c -> c.getChild("enumeration", false))
                    .map(c -> c.getChild("custom-enumerator", false))
                    .orElse(null);
                if (customEnumConf.getChild("schedulable", false) == null)
                {
                    DefaultConfiguration schedulableConf = (DefaultConfiguration) customEnumConf.getChild("schedulable");
                    schedulableConf.setValue(this.getClass().getName());
                    ((DefaultConfiguration) customEnumConf).addChild(schedulableConf);
                }
                break;
            }
        }
        super.configure(configuration);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _reportEP = (ReportExtensionPoint) manager.lookup(ReportExtensionPoint.ROLE);
        _reportHelper = (ReportHelper) manager.lookup(ReportHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        String extensionId = jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_EXTENSION_ID_KEY);
        PilotageReport report = _reportEP.getExtension(extensionId);
        
        // Prepare ZIP creation
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute("report", report);
        request.setAttribute("reportTarget", getTarget());
        request.setAttribute("reportParameters", getReportParameters(jobDataMap));
        
        Source source = null;
        String status = null;
        File file = null;
        try
        {
            // Get the ZIP file content
            source = _sourceResolver.resolveURI("cocoon://_plugins/odf-pilotage/pilotage-report.zip");
            try (InputStream is = source.getInputStream())
            {
                status = (String) request.getAttribute("reportStatus");
                if (status == null)
                {
                    String zipFileName = (String) request.getAttribute("reportZipName");
                    
                    // Delete existing file
                    file = new File(_reportHelper.getPilotageFolder(), zipFileName);
                    FileUtils.deleteQuietly(file);
                    file.createNewFile();
    
                    // Save file
                    try (OutputStream os = new FileOutputStream(file))
                    {
                        IOUtils.copy(is, os);
                    }
                    
                    status = "SUCCESS";
                }
            }
        }
        catch (Exception e)
        {
            status = "FAIL";
            throw e;
        }
        finally
        {
            if (source != null)
            {
                _sourceResolver.release(source);
            }
            
            _sendMail(status, report.getLabel(), file);
        }
    }
    
    /**
     * The target of the report.
     * @return The target
     */
    public abstract PilotageReportTarget getTarget();
    
    /**
     * The schedulable is written for generic reports or it has additional parameters.
     * @return <code>true</code> if the schedulable is only for generic reports.
     */
    public boolean forGenericReports()
    {
        return true;
    }
    
    /**
     * Get the report parameters from the schedulable parameters
     * @param jobDataMap The schedulable parameters
     * @return The report parameters
     */
    protected Map<String, String> getReportParameters(JobDataMap jobDataMap)
    {
        Map<String, String> reportParameters = new HashMap<>();
        reportParameters.put(PilotageReport.PARAMETER_OUTPUT_FORMAT, jobDataMap.getString(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_OUTPUT_FORMAT_KEY));
        return reportParameters;
    }
    
    /**
     * Send a mail with the ZIP file as attachment at the end of the report generation.
     * @param status the status of the report generation
     * @param reportLabel the label of the report
     * @param zipFile the pilotage file
     */
    protected void _sendMail(String status, I18nizableText reportLabel, File zipFile)
    {
        String body = _i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MAIL_BODY_" + status, Map.of("name", reportLabel)));
        getLogger().info(body);
        
        String recipient = Optional.ofNullable(_currentUserProvider.getUser())
            .map(_userManager::getUser)
            .map(User::getEmail)
            .filter(StringUtils::isNotEmpty)
            .orElse(null);
        
        if (recipient != null)
        {
            String subject = _i18nUtils.translate(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_MAIL_SUBJECT", Map.of("name", reportLabel)));
            try
            {
                List<File> attachments = Optional.ofNullable(zipFile)
                        .filter(File::exists)
                        .stream()
                        .toList();
                
                getLogger().info("Envoi du rapport par mail");
                SendMailHelper.newMail()
                              .withSubject(subject)
                              .withTextBody(body)
                              .withAttachments(attachments)
                              .withRecipient(recipient)
                              .withSender(_mailFrom)
                              .sendMail();
            }
            catch (MessagingException | IOException e)
            {
                getLogger().warn("Fail to send email to {}", recipient, e);
            }
        }
    }
}
