/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tool;

import java.io.File;
import java.io.FileFilter;
import java.time.ZonedDateTime;

import org.ametys.core.util.DateUtils;

/**
 * File filter for pilotage files and search of these files.
 */
public class PilotageFileFilter implements FileFilter
{
    private ZonedDateTime _lastModifiedAfter;
    private ZonedDateTime _lastModifiedBefore;
    private String _fileName;
    
    /**
     * Constructor
     * @param fileName Filter on the filename
     * @param lastModifiedAfter Filter on the date after
     * @param lastModifiedBefore Filter on the date before
     */
    public PilotageFileFilter(String fileName, ZonedDateTime lastModifiedAfter, ZonedDateTime lastModifiedBefore)
    {
        _fileName = fileName;
        _lastModifiedAfter = lastModifiedAfter;
        _lastModifiedBefore = lastModifiedBefore;
    }
    
    @Override
    public boolean accept(File pathname)
    {
        /* On accepte que les fichiers */
        if (pathname.isDirectory())
        {
            return false;
        }
        
        /* Tests sur le nom de fichier */
        String fileName = pathname.getName().toLowerCase();
        
        if (!fileName.matches(".*\\.zip"))
        {
            return false;
        }
        if (_fileName != null && !fileName.contains(_fileName.toLowerCase()))
        {
            return false;
        }

        /* Tests sur la date de dernière modification */
        ZonedDateTime lastModified = DateUtils.asZonedDateTime(pathname.lastModified());
        if (_lastModifiedAfter != null && lastModified.isBefore(_lastModifiedAfter))
        {
            return false;
        }
        if (_lastModifiedBefore != null && lastModified.isAfter(_lastModifiedBefore))
        {
            return false;
        }
        
        return true;
    }
}
