/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.validator;

import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.contenttype.validation.AbstractContentValidator;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.View;
import org.ametys.runtime.parameter.ValidationResult;

/**
 * Validates that the mcc regime is filled if the regime policy is set to FORCE for containers with year nature.
 */
public class ContainerMccRegimeValidator extends AbstractContentValidator implements Serviceable
{
    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    private boolean _isForceMode;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
    }
    
    @Override
    public void initialize() throws ConfigurationException
    {
        super.initialize();
        _isForceMode = _pilotageHelper.getMCCRegimePolicy().equals("FORCE");
    }
    
    public ValidationResult validate(Content content)
    {
        ValidationResult result = new ValidationResult();
        
        // If:
        //  - regime policy is FORCE
        //  - content is a container
        //  - container nature is year
        //  - mccRegime field is empty or value does not exist anymore
        // Then add an error
        if (
                _isForceMode
                && _pilotageHelper.isContainerOfTypeYear(content)
                && Optional.of(PilotageHelper.CONTAINER_MCC_REGIME)
                    .filter(content::hasValue)
                    .map(content::<ContentValue>getValue)
                    .flatMap(ContentValue::getContentIfExists)
                    .isEmpty()
            )
        {
            result.addError(
                new I18nizableText(
                    "plugin.odf-pilotage",
                    "PLUGINS_ODF_PILOTAGE_CONTAINER_MCCREGIME_VALIDATOR_ERROR",
                    Map.of(
                        "fieldRegime", content.getDefinition(PilotageHelper.CONTAINER_MCC_REGIME).getLabel(),
                        "forceValue", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CONFIG_MCC_RESTRICTIONS_POLICY_FORCE")
                    )
                )
            );
        }
        
        return result;
    }

    public ValidationResult validate(Content content, Map<String, Object> values, View view)
    {
        ValidationResult result = new ValidationResult();
        
        // If:
        //  - regime policy is FORCE
        //  - content is a container
        //  - mccRegime is in the form
        if (_isForceMode && content instanceof Container container && view.hasModelViewItem(PilotageHelper.CONTAINER_MCC_REGIME))
        {
            // If:
            //  - mccRegime is not an UntouchedValue (cannot be modified)
            //  - mccRegime field is empty or value does not exist anymore
            if (!(values.get(PilotageHelper.CONTAINER_MCC_REGIME) instanceof UntouchedValue)
                && Optional.of(PilotageHelper.CONTAINER_MCC_REGIME)
                    .map(values::get)
                    .map(SynchronizableValue.class::cast)
                    .map(sv -> sv.getValue(Optional.ofNullable(sv.getExternalizableStatus())))
                    .map(ContentValue.class::cast)
                    .flatMap(ContentValue::getContentIfExists)
                    .map(Content::getId)
                    .isEmpty())
            {
                // If the nature can be modified by the form, read the container nature from the form
                // Otherwise, read the container nature from the content
                String nature = Optional.of("nature")
                        .map(values::get)
                        .filter(Predicate.not(UntouchedValue.class::isInstance))
                        .map(SynchronizableValue.class::cast)
                        .map(sv -> sv.getValue(Optional.ofNullable(sv.getExternalizableStatus())))
                        .map(ContentValue.class::cast)
                        .map(ContentValue::getContentId)
                        .orElseGet(() -> container.getNature());
                
                // If the container nature is year
                // Then add an error
                if (nature != null && _pilotageHelper.getYearId().map(nature::equals).orElse(false))
                {
                    result.addError(
                        new I18nizableText(
                            "plugin.odf-pilotage",
                            "PLUGINS_ODF_PILOTAGE_CONTAINER_MCCREGIME_VALIDATOR_ERROR",
                            Map.of(
                                "fieldRegime", content.getDefinition(PilotageHelper.CONTAINER_MCC_REGIME).getLabel(),
                                "forceValue", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CONFIG_MCC_RESTRICTIONS_POLICY_FORCE")
                            )
                        )
                    );
                }
            }
        }
        
        return result;
    }
}
