/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.clientsideelement;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.core.ui.Callable;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.clientsideelement.DeleteContentClientSideElement;
import org.ametys.plugins.odfpilotage.helper.PilotageClientSideElementHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;

/**
 * Override button for odf content to take into account the attribute restrictions on parent
 */
public class DeleteODFPilotageContentClientSideElement extends DeleteContentClientSideElement
{
    /** The pilotage status helper */
    protected PilotageClientSideElementHelper _pilotageClientSideHelper;
    private ODFHelper _odfHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _pilotageClientSideHelper = (PilotageClientSideElementHelper) smanager.lookup(PilotageClientSideElementHelper.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    @Callable
    @Override
    public Map<String, Object> getStatus(List<String> contentIds)
    {
        Map<String, Object> results = super.getStatus(contentIds);
        
        List<Map<String, Object>> invalidContents = new ArrayList<>();
                
        for (String contentId : contentIds)
        {
            Content content = _resolver.resolveById(contentId);
            if (content instanceof ProgramItem programItem)
            {
                List<ProgramItem> parentProgramItems = _odfHelper.getParentProgramItems(programItem);
                for (ProgramItem parentProgramItem : parentProgramItems)
                {
                    String parentAttributeName = _odfHelper.getDescendantRelationAttributeName(parentProgramItem, programItem);
                    if (parentAttributeName != null)
                    {
                        ModelItem modelItem = ModelHelper.getModelItem(parentAttributeName, ((Content) parentProgramItem).getModel());
                        if (modelItem instanceof RestrictedModelItem rModelItem && !rModelItem.canWrite(parentProgramItem))
                        {
                            Map<String, Object> contentParams = getContentDefaultParameters (content);
                            contentParams.put("description", _getInvalidRestrictionStatusDescription(content));
                            invalidContents.add(contentParams);
                            
                            // At least on parent does not allow to delete content: stop iteration on parent
                            break;
                        }
                    }
                }
            }
        }
        
        List<Object> invalidContentIds = invalidContents.stream()
                .map(p -> p.get("id"))
                .toList();
        
        // Remove content with invalid MCC status from all right contents
        List<Map<String, Object>> allrightContents = (List<Map<String, Object>>) results.get("allright-contents");
        results.put("allright-contents", allrightContents.stream().filter(p -> !invalidContentIds.contains(p.get("id"))).toList());
        results.put("invalidrestrictionstatus-contents", invalidContents);
        
        return results;
    }
    
    /**
     * Get i18n description when the restrcition does not allow edition
     * @param content The content
     * @return The {@link I18nizableText} description
     */
    protected I18nizableText _getInvalidRestrictionStatusDescription (Content content)
    {
        List<String> workflowI18nParameters = new ArrayList<>();
        workflowI18nParameters.add(_contentHelper.getTitle(content));
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("invalidrestrictionstatus-content-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), workflowI18nParameters);
    }
}
