/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report;

import java.text.Normalizer;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.ReportHelper;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * The abstract class for pilotage reports.
 */
public abstract class AbstractPilotageReport extends AbstractLogEnabled implements PilotageReport, Serviceable, PluginAware, Configurable
{
    /** Filename of the manifest to describe the ZIP content */
    public static final String MANIFEST_FILENAME = "manifest.json";
    
    private static final int __REPORT_FILENAME_MAX_LENGTH = 120;

    /** The pilotage helper */
    protected PilotageHelper _pilotageHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The report helper */
    protected ReportHelper _reportHelper;
    
    /** The ODF enumeration helper */
    protected OdfReferenceTableHelper _refTableHelper;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    private String _id;
    private I18nizableText _label;
    private String _pluginName;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _id = id;
    }
    
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = I18nizableText.parseI18nizableText(configuration.getChild("label"), "plugin." + _pluginName);
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _reportHelper = (ReportHelper) manager.lookup(ReportHelper.ROLE);
        _refTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
    }

    public String getId()
    {
        return _id;
    }
    
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    public boolean supports(AbstractReportSchedulable schedulable)
    {
        if (schedulable.forGenericReports() == isGeneric() && isSupportedTarget(schedulable.getTarget()))
        {
            return isCompatibleSchedulable(schedulable);
        }
        
        return false;
    }
    
    /**
     * Most of reports are generic. This method can be overridden.
     * @return <code>true</code> if the current report is generic, <code>false</code> otherwise
     */
    protected boolean isGeneric()
    {
        return true;
    }
    
    /**
     * Returns <code>true</code> if the target is supported by the report.
     * @param target The target to test
     * @return <code>true</code> if the target is supported, <code>false</code> otherwise
     */
    protected abstract boolean isSupportedTarget(PilotageReportTarget target);

    /**
     * Check if the given schedulable is compatible with the current
     * @param schedulable The schedulable to test
     * @return <code>true</code> if the schedulable is compatible with the report
     */
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return true;
    }
    
    /**
     * Launch a report generation on an orgunit.
     * @param outputFormat The output format
     * @param reportParameters The report parameters
     * @return the name of the generated file
     */
    protected abstract PilotageReportContent getReportContentForOrgUnit(String outputFormat, Map<String, String> reportParameters);

    /**
     * Launch a report generation on a program.
     * @param outputFormat The output format
     * @param reportParameters The report parameters
     * @return the name of the generated file
     */
    protected abstract PilotageReportContent getReportContentForProgram(String outputFormat, Map<String, String> reportParameters);

    public String getType()
    {
        return getType(null);
    }
    
    /**
     * Get the name of the report with complements if report parameters is filled.
     * @param reportParameters The report parameters
     * @return The report name
     */
    protected abstract String getType(Map<String, String> reportParameters);
    
    public String getPluginName()
    {
        return _pluginName;
    }
    
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_DOC;
    }
    
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_DOC, OUTPUT_FORMAT_XLS);
    }

    public boolean isSupportedFormat(String outputFormat)
    {
        return getSupportedOutputFormats().contains(outputFormat);
    }
    
    public PilotageReportContent getReportContent(PilotageReportTarget target, Map<String, String> reportParameters)
    {
        getLogger().info("Début du rapport de pilotage");
        long begin = System.currentTimeMillis();

        try
        {
            // Check the output format
            String outputFormat = reportParameters.get(PARAMETER_OUTPUT_FORMAT);
            if (!isSupportedFormat(outputFormat))
            {
                throw new UnsupportedOperationException("Impossible to launch the report '" + getType() + "' with the output format '" + outputFormat + "'.");
            }
            
            // Check the target
            if (!isSupportedTarget(target))
            {
                throw new UnsupportedOperationException("Impossible to launche the report '" + getType() + "' on the target '" + target.name() + "'");
            }
            
            switch (target)
            {
                case PROGRAM:
                    return getReportContentForProgram(outputFormat, reportParameters);
                case ORGUNIT:
                    return getReportContentForOrgUnit(outputFormat, reportParameters);
                default:
                    throw new UnsupportedOperationException("Not supported target '" + target + "'.");
            }
        }
        finally
        {
            long end = System.currentTimeMillis();
            getLogger().info("Calcul et écriture du rapport de pilotage effectué en {} ms.", end - begin);
        }
    }
    
    public String getReportFileName(String catalog, String lang, OrgUnit orgUnit, Map<String, String> reportParameters, String outputFormat)
    {
        StringBuilder sb = new StringBuilder();
        
        // Type
        sb.append(getType(reportParameters));
        sb.append("-");
        
        // Catalog
        sb.append(catalog);
        sb.append("-");
        
        // Lang
        sb.append(lang);
        sb.append("-");
        
        // Acronym ou code UAI
        sb.append(_reportHelper.getAcronymOrUaiCode(orgUnit));
        sb.append("-");
        
        // Date
        sb.append(_getCurrentDate());
        
        // Output format
        sb.append(".");
        sb.append(outputFormat);

        return sb.toString();
    }
    
    public String getReportFileName(ProgramItem programItem, Map<String, String> reportParameters, String outputFormat)
    {
        StringBuilder filenamePrefix = new StringBuilder();
        
        // Type
        filenamePrefix.append(getType(reportParameters));
        filenamePrefix.append("-");
        
        // Catalog
        filenamePrefix.append(programItem.getCatalog());
        filenamePrefix.append("-");
        
        // Lang
        filenamePrefix.append(programItem.getLanguage());
        filenamePrefix.append("-");
        
        StringBuilder filenameSuffix = new StringBuilder();
        filenameSuffix.append("-");
        
        // Code
        filenameSuffix.append(programItem.getCode());
        filenameSuffix.append("-");
        
        // Date
        filenameSuffix.append(_getCurrentDate());
        
        // Title - truncate to respect the filename max length
        int titleMaxLength = __REPORT_FILENAME_MAX_LENGTH - (filenamePrefix.length() + filenameSuffix.length());
        String title = StringUtils.substring(((Content) programItem).getTitle(), 0, titleMaxLength);
        
        // Concat file name segments
        StringBuilder filename = filenamePrefix.append(title)
                                               .append(filenameSuffix);
        
        // Normalize and add output format
        return _normalizeFileName(filename.toString()) + "." + outputFormat;
    }
    
    private String _getCurrentDate()
    {
        return DateUtils.localDateToString(LocalDate.now());
    }
    
    /**
     * Get the report content for the program
     * @param outputFormat the output format
     * @param reportParameters the report parameters
     * @return the report content
     */
    protected PilotageReportContent _getReportContentForProgram(String outputFormat, Map<String, String> reportParameters)
    {
        String programId = reportParameters.get(PARAMETER_PROGRAM);
        ProgramItem programItem =  _resolver.resolveById(programId);
        if (!(programItem instanceof Program program))
        {
            throw new UnsupportedOperationException("The report '" + getType() + "' can be launch only on programs through this method.");
        }
        
        return new PilotageReportContent(
            _buildZipName(reportParameters),
            _getFilesList(_getProgramItemsFromProgram(program, reportParameters), _getProgramItemTransform(outputFormat, reportParameters))
        );
    }
    
    /**
     * Get the report content for programs of an org unit
     * @param outputFormat the output format
     * @param reportParameters the report parameters
     * @return the report content
     */
    protected PilotageReportContent _getReportContentForProgramItemsInOrgUnit(String outputFormat, Map<String, String> reportParameters)
    {
        String orgunitId = reportParameters.get(PARAMETER_ORGUNIT);
        String catalog = reportParameters.get(PARAMETER_CATALOG);
        String lang = reportParameters.get(PARAMETER_LANG);
        
        OrgUnit orgUnit = null;
        if (StringUtils.isNotEmpty(orgunitId))
        {
            orgUnit = _resolver.resolveById(orgunitId);
        }
        
        return new PilotageReportContent(
            _buildZipName(reportParameters),
            _getFilesList(_getProgramItemsFromOrgUnit(orgUnit, catalog, lang, reportParameters), _getProgramItemTransform(outputFormat, reportParameters))
        );
    }
    
    /**
     * Get the transformation from program item to pair of filename and pipeline to generate the file. Can be null.
     * @param outputFormat The output format
     * @param reportParameters The report parameters
     * @return a pair of filename and pipeline
     */
    protected Function<ProgramItem, Pair<String, String>> _getProgramItemTransform(String outputFormat, Map<String, String> reportParameters)
    {
        return programItem ->
            Pair.of(
                getReportFileName(programItem, reportParameters, outputFormat),
                _buildPipeline(outputFormat, reportParameters, "programItem", programItem)
            );
    }
    
    private Stream<? extends ProgramItem> _getProgramItemsFromOrgUnit(OrgUnit orgUnit, String catalog, String lang, Map<String, String> reportParameters)
    {
        return _odfHelper.getProgramsFromOrgUnit(orgUnit, catalog, lang).stream()
            .flatMap(p -> _getProgramItemsFromProgram(p, reportParameters))
            .distinct();
    }
    
    /**
     * The program items from the program.
     * @param program The program
     * @param reportParameters The report parameters
     * @return A stream of program items
     */
    protected Stream<? extends ProgramItem> _getProgramItemsFromProgram(Program program, Map<String, String> reportParameters)
    {
        return Stream.of(program);
    }
    
    /**
     * Get the report content foran org unit
     * @param outputFormat the output format
     * @param reportParameters the report parameters
     * @return the report content
     */
    protected PilotageReportContent _getReportContentForOrgUnit(String outputFormat, Map<String, String> reportParameters)
    {
        String orgunitId = reportParameters.get(PARAMETER_ORGUNIT);
        String catalog = reportParameters.get(PARAMETER_CATALOG);
        String lang = reportParameters.get(PARAMETER_LANG);
        
        List<OrgUnit> orgUnits = _reportHelper.getOrgUnits(orgunitId);
        Function<OrgUnit, Pair<String, String>> transform = orgUnit ->
            Pair.of(
                getReportFileName(catalog, lang, orgUnit, reportParameters, outputFormat),
                _buildPipeline(outputFormat, reportParameters, "orgUnit", orgUnit)
            );
        
        return new PilotageReportContent(
            _buildZipName(reportParameters),
            _getFilesList(orgUnits.stream(), transform)
        );
    }
    
    private <T> Map<String, String> _getFilesList(Stream<? extends T> elements, Function<T, Pair<String, String>> transform)
    {
        return elements
            .distinct()
            .map(transform::apply)
            .filter(Objects::nonNull)
            .collect(Collectors.toMap(Pair::getKey, Pair::getValue));
    }
    
    /**
     * Build the pipeline to generate the report.
     * @param outputFormat The output format
     * @param reportParameters The report parameters
     * @param contentParameterName The content parameter name
     * @param content The content
     * @return the pipeline
     */
    protected String _buildPipeline(String outputFormat, Map<String, String> reportParameters, String contentParameterName, AmetysObject content)
    {
        StringBuilder sb = new StringBuilder();
        sb.append("cocoon://_plugins/odf-pilotage/report/");
        sb.append(getType());
        sb.append(".");
        sb.append(outputFormat);
        sb.append("?");
        sb.append(contentParameterName);
        sb.append("=");
        sb.append(URIUtils.encodeParameter(content.getId()));
        return sb.toString();
    }
    
    /**
     * Build the ZIP name.
     * @param reportParameters The report parameters
     * @return The full ZIP name
     */
    private String _buildZipName(Map<String, String> reportParameters)
    {
        return new StringBuilder()
                .append(getType(reportParameters))
                .append("-")
                .append(org.ametys.core.util.StringUtils.generateKey())
                .append(".zip")
                .toString();
    }
    
    public void saxOrgUnit(ContentHandler handler, String orgUnitId, Map<String, String> reportParameters)
    {
        _saxOrgUnit(handler, reportParameters.get(PARAMETER_CATALOG), reportParameters.get(PARAMETER_LANG), orgUnitId, reportParameters);
    }
    
    /**
     * Sax an org unit on the given content handler from the orgunit identifier and report parameters
     * @param handler The handler
     * @param catalog The catalog
     * @param lang The language
     * @param orgUnitId The orgunit identifier
     * @param reportParameters The report parameters
     */
    protected abstract void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters);
    
    private String _normalizeFileName(String originalName)
    {
        // Use lower case
        // then remove accents
        // then replace contiguous spaces with one dash
        // and finally remove non-alphanumeric characters except -
        String filteredName = Normalizer.normalize(originalName.toLowerCase(), Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "").trim();
        filteredName = filteredName.replaceAll("œ", "oe").replaceAll("æ", "ae").replaceAll(" +", "-").replaceAll("[^\\w-]", "-").replaceAll("-+", "-");
        
        // Remove characters '-' and '_' at the start and the end of the string
        return org.apache.commons.lang3.StringUtils.strip(filteredName, "-_");
    }
}
