/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.URIUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.report.impl.mcc.MCCProgramItemTree;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.odfpilotage.rule.export.SetMCCHeaderAction;
import org.ametys.plugins.odfpilotage.schedulable.AbstractReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.OrgUnitMCCReportSchedulable;
import org.ametys.plugins.odfpilotage.schedulable.ProgramMCCReportSchedulable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;

/**
 * Class to generate the MCC report.
 */
public class MCCReport extends AbstractMCCReport
{
    /** The key for the definitive versions */
    public static final String PARAMETER_DEFINITIVE = "definitive";
    
    /** The key for the alternatives versions */
    public static final String PARAMETER_RULES_VERSION = "rulesVersion";

    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
    }
    
    @Override
    protected boolean isCompatibleSchedulable(AbstractReportSchedulable schedulable)
    {
        return schedulable instanceof ProgramMCCReportSchedulable || schedulable instanceof OrgUnitMCCReportSchedulable;
    }
    
    @Override
    protected boolean isSupportedTarget(PilotageReportTarget target)
    {
        return super.isSupportedTarget(target) || PilotageReportTarget.PROGRAM.equals(target);
    }
    
    @Override
    protected PilotageReportContent getReportContentForProgram(String outputFormat, Map<String, String> reportParameters)
    {
        return _getReportContentForProgram(outputFormat, reportParameters);
    }
    
    @Override
    protected String getType(Map<String, String> reportParameters)
    {
        String type = "mcc";
        
        if (reportParameters != null)
        {
            if (Boolean.parseBoolean(reportParameters.get(PARAMETER_DEFINITIVE)))
            {
                type += "-definitif";
            }
            else if (RulesManager.isRulesEnabled() && !reportParameters.get(PARAMETER_OUTPUT_FORMAT).equals(OUTPUT_FORMAT_XLS))
            {
                String rulesVersion = reportParameters.getOrDefault(PARAMETER_RULES_VERSION, "final");
                type += SetMCCHeaderAction.VERSION_TO_SUFFIX.getOrDefault(rulesVersion, "-" + rulesVersion);
            }
        }
        
        return type;
    }
    
    @Override
    protected void _saxMCCs(ContentHandler handler, Course course, MCCProgramItemTree tree) throws SAXException
    {
        XMLUtils.startElement(handler, "mcc");
        
        // Generate SAX events for MCC sessions
        _saxSession(handler, course, FIRST_SESSION_NAME);
        _saxSession(handler, course, SECOND_SESSION_NAME);
        
        XMLUtils.endElement(handler, "mcc");
    }
    
    private void _saxSession(ContentHandler handler, Course course, String sessionName) throws SAXException
    {
        if (course.hasValue(sessionName) && !_disableConditionsEvaluator.evaluateDisableConditions(_sessionsView.getModelViewItem(sessionName).getDefinition(), sessionName, course))
        {
            XMLUtils.startElement(handler, sessionName);
            
            ModelAwareRepeater session = course.getRepeater(sessionName);
            for (ModelAwareRepeaterEntry sessionEntry : session.getEntries())
            {
                _saxSessionEntry(handler, sessionEntry, course);
            }
            
            XMLUtils.endElement(handler, sessionName);
        }
    }
    
    /**
     * Sax a MCC session entry.
     * @param handler The handler
     * @param sessionEntry The session entry name
     * @throws SAXException If an error occurs
     */
    private void _saxSessionEntry(ContentHandler handler, ModelAwareRepeaterEntry sessionEntry, Content rootContent) throws SAXException
    {
        // start entry
        AttributesImpl entryAttrs = new AttributesImpl();
        entryAttrs.addCDATAAttribute("name", String.valueOf(sessionEntry.getPosition()));
        XMLUtils.startElement(handler, "entry", entryAttrs);
        
        _saxSessionEntryDetails(handler, sessionEntry, rootContent);
        
        // end entry
        XMLUtils.endElement(handler, "entry");
    }
    
    @Override
    protected void populateMCCTree(MCCProgramItemTree tree)
    {
        ProgramItem programItem = tree.getCurrent();
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem child : children)
        {
            MCCProgramItemTree childTree = tree.addChild(child);
            populateMCCTree(childTree);
        }
    }
    
    @Override
    protected Function<ProgramItem, Pair<String, String>> _getProgramItemTransform(String outputFormat, Map<String, String> reportParameters)
    {
        return Boolean.parseBoolean(reportParameters.get(PARAMETER_DEFINITIVE))
            // We want to get archived versions
            ? programItem -> {
                // The content have to be a container
                if (!(programItem instanceof Container container))
                {
                    throw new UnsupportedOperationException("The content '" + programItem.getId() + "' is not a Container and cannot be managed.");
                }
                
                ModelAwareRepeaterEntry entry = _mccWorkflowHelper.getLastMCCValidatedEntry(container);
                if (entry != null)
                {
                    String attributePath = entry.getHoldingRepeater().getModel().getName() + "[" + entry.getPosition() + "]/pdf";
                    String uri = attributePath + "?contentId=" + programItem.getId();
                    String fileUri = ResolveURIComponent.resolve("attribute", uri, false, false, true);
                    if (StringUtils.isNotEmpty(fileUri))
                    {
                        return Pair.of(entry.<Binary>getValue("pdf").getFilename(), fileUri);
                    }
                }
                
                // If there is no file, return null
                return null;
            }
            // Other cases
            : super._getProgramItemTransform(outputFormat, reportParameters);
    }
    
    @Override
    protected String _buildPipeline(String outputFormat, Map<String, String> reportParameters, String contentParameterName, AmetysObject content)
    {
        // If rules are enabled, generate the file with rules
        // Except for XLS format which is not supported with rules
        if (RulesManager.isRulesEnabled() && !outputFormat.equals(OUTPUT_FORMAT_XLS))
        {
            StringBuilder sb = new StringBuilder();
            sb.append("cocoon://_plugins/odf-pilotage/");
            sb.append(reportParameters.getOrDefault(PARAMETER_RULES_VERSION, "final"));
            sb.append("/container-rules.");
            sb.append(outputFormat);
            sb.append("?");
            sb.append(contentParameterName);
            sb.append("=");
            sb.append(URIUtils.encodeParameter(content.getId()));
            sb.append("&reportId=");
            sb.append(this.getId());
            return sb.toString();
        }
        
        // We want the "classic" MCC report file
        return super._buildPipeline(outputFormat, reportParameters, contentParameterName, content);
    }
}
