/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.pipeline;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.util.URIUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.plugins.odfpilotage.report.PilotageReport;
import org.ametys.plugins.odfpilotage.report.ReportExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Action to set report parameters and response header before launching it.
 */
public class SetReportParametersAndHeaderAction extends ServiceableAction
{
    /** The prefix key for parameters */
    public static final String PARAMETER_PREFIX = SetReportParametersAndHeaderAction.class.getName() + "$";
    
    /** The report extension point */
    protected ReportExtensionPoint _reportEP;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _reportEP = (ReportExtensionPoint) smanager.lookup(ReportExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // Set report EP to request attributes
        String reportId = request.getParameter("reportId");
        PilotageReport report = _reportEP.getExtension(reportId);
        request.setAttribute("report", report);

        // Check the output format
        String outputFormat = parameters.getParameter(PilotageReport.PARAMETER_OUTPUT_FORMAT);
        if (!report.isSupportedFormat(outputFormat))
        {
            throw new UnsupportedOperationException("Impossible to launch the report '" + report.getType() + "' with the output format '" + outputFormat + "'.");
        }
        
        // Set report plugin name
        request.setAttribute("reportPluginName", report.getPluginName());
        
        // Set report parameters to request attributes
        Map<String, String> reportParameters = new HashMap<>();
        reportParameters.put(PilotageReport.PARAMETER_OUTPUT_FORMAT, outputFormat);
        request.setAttribute("reportParameters", reportParameters);
        
        // Configure the response header
        Response response = ObjectModelHelper.getResponse(objectModel);
        ProgramItem programItem = _resolver.resolveById(request.getParameter("programItem"));
        String fileName = URIUtils.encodeHeader(report.getReportFileName(programItem, reportParameters, outputFormat));
        response.setHeader("Content-Disposition", "attachment;filename=\"" + fileName + "\";filename*=UTF-8''" + fileName);
        
        return Map.of("reportType", report.getType());
    }
}
